<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Facebook Pixel Base for Magento 2
 */

namespace Amasty\FacebookPixel\Controller\Adminhtml\Pixels;

use Amasty\FacebookPixel\Api\Data\PixelInterface;
use Amasty\FacebookPixel\Api\PixelRepositoryInterface;
use Amasty\FacebookPixel\Model\ResourceModel\Pixel\CollectionFactory;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Phrase;
use Magento\Ui\Component\MassAction\Filter;
use Psr\Log\LoggerInterface;

class MassDisable extends Action
{
    const ADMIN_RESOURCE = 'Amasty_FacebookPixel::manage_pixels';

    /**
     * @var PixelRepositoryInterface
     */
    private $pixelRepository;

    /**
     * @var Filter
     */
    private $filter;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var LoggerInterface
     */
    private $logger;

    public function __construct(
        Context $context,
        PixelRepositoryInterface $pixelRepository,
        Filter $filter,
        CollectionFactory $collectionFactory,
        LoggerInterface $logger
    ) {
        parent::__construct($context);
        $this->pixelRepository = $pixelRepository;
        $this->logger = $logger;
        $this->filter = $filter;
        $this->collectionFactory = $collectionFactory;
    }

    /**
     * @return ResultInterface
     */
    public function execute(): ResultInterface
    {
        $collection = $this->collectionFactory->create();
        $collection = $this->filter->getCollection($collection);
        $collectionSize = $collection->getSize();

        if ($collectionSize) {
            try {
                $updatedCount = 0;

                foreach ($collection->getItems() as $model) {
                    /** @var PixelInterface $model */
                    try {
                        $model->setIsActive(false);
                        $this->pixelRepository->save($model);
                        $updatedCount++;
                    } catch (LocalizedException $e) {
                        $this->messageManager->addErrorMessage($e->getMessage());
                    }
                }

                $this->messageManager->addSuccessMessage($this->getSuccessMessage($updatedCount));
            } catch (\Exception $e) {
                $this->messageManager->addErrorMessage($this->getErrorMessage());
                $this->logger->critical($e);
            }
        }

        return $this->resultRedirectFactory->create()->setUrl($this->_redirect->getRefererUrl());
    }

    /**
     * @return Phrase
     */
    private function getErrorMessage(): Phrase
    {
        return __('We can\'t save item right now. Please review the log and try again.');
    }

    /**
     * @param int $collectionSize
     *
     * @return Phrase
     */
    private function getSuccessMessage(int $collectionSize = 0): Phrase
    {
        if ($collectionSize) {
            return __('A total of %1 record(s) have been disabled.', $collectionSize);
        }

        return __('No records have been changed.');
    }
}
