<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Dropshipping for Magento 2
 */

namespace Amasty\Dropshipping\Setup\Patch\Data;

use Amasty\Dropshipping\Model\Config\Source\Email\Template\Assign;
use Amasty\Dropshipping\Model\ConfigProvider;
use Magento\Email\Model\ResourceModel\Template as TemplateResource;
use Magento\Email\Model\ResourceModel\Template\Collection as TemplateCollection;
use Magento\Email\Model\ResourceModel\Template\CollectionFactory as TemplateCollectionFactory;
use Magento\Email\Model\Template;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Setup\Patch\DataPatchInterface;
use Magento\Framework\Stdlib\DateTime;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;

class CopyTemplates implements DataPatchInterface
{
    public const ORDER_ASSIGN_TEMPLATE_ORIG_CODE = 'amasty_perm_sales_email_order_assign_template';
    public const ORIG_TEMPLATE_CODE = 'orig_template_code';
    public const OLD_CONFIG_PATH = 'amasty_dropshipping/messages/template';
    public const NEW_CONFIG_PATH = 'amasty_dropshipping/' . ConfigProvider::MESSAGES_ASSIGN_TEMPLATE;

    /**
     * @var TemplateCollectionFactory
     */
    private $templateCollectionFactory;

    /**
     * @var TemplateResource
     */
    private $templateResource;

    /**
     * @var WriterInterface
     */
    private $configWriter;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var TimezoneInterface
     */
    private $timezone;

    public function __construct(
        TemplateCollectionFactory $templateCollectionFactory,
        TemplateResource $templateResource,
        ScopeConfigInterface $scopeConfig,
        WriterInterface $configWriter,
        TimezoneInterface $timezone
    ) {
        $this->templateCollectionFactory = $templateCollectionFactory;
        $this->templateResource = $templateResource;
        $this->scopeConfig = $scopeConfig;
        $this->configWriter = $configWriter;
        $this->timezone = $timezone;
    }

    public static function getDependencies()
    {
        return [];
    }

    public function getAliases()
    {
        return [];
    }

    public function apply()
    {
        $oldTemplateId = $this->scopeConfig->getValue(self::OLD_CONFIG_PATH);

        /** @var TemplateCollection $templateCollection */
        $templateCollection = $this->templateCollectionFactory->create();

        $templateCollection->addFieldToFilter(
            self::ORIG_TEMPLATE_CODE,
            self::ORDER_ASSIGN_TEMPLATE_ORIG_CODE
        );

        /** @var Template $template */
        foreach ($templateCollection as $template) {
            $templateId = $template->getId();

            $templateCode = sprintf(
                '%s [%s]',
                $template->getTemplateCode(),
                $this->timezone->date()->format(DateTime::DATETIME_PHP_FORMAT)
            );

            $template->setId(null)
                ->setTemplateCode($templateCode)
                ->setData(self::ORIG_TEMPLATE_CODE, Assign::TEMPLATE_ID);

            $this->templateResource->save($template);

            if (is_numeric($oldTemplateId) && $oldTemplateId == $templateId) {
                $this->configWriter->save(self::NEW_CONFIG_PATH, $template->getId());
            }
        }
    }
}
