<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Dropshipping for Magento 2
 */
namespace Amasty\Dropshipping\Plugin;

use Amasty\Dropshipping\Api\Data\DealerItemInterfaceFactory;
use Amasty\Dropshipping\Helper\Data as PermHelper;
use Amasty\Dropshipping\Model\Dealer;
use Amasty\Dropshipping\Model\DealerItem\DealerItemRepository;
use Amasty\Dropshipping\Model\DealerOrder;
use Amasty\Dropshipping\Model\DealerOrderFactory;
use Magento\Sales\Api\Data\OrderInterface;

class OrderService
{
    /**
     * @var DealerItemInterfaceFactory
     */
    private $dealerItemInterfaceFactory;

    /**
     * @var DealerOrderFactory
     */
    protected $_dealerOrderFactory;

    /**
     * @var PermHelper
     */
    protected $_permHelper;

    /**
     * @var DealerItemRepository
     */
    private $dealerItemRepository;

    public function __construct(
        DealerItemInterfaceFactory $dealerItemInterfaceFactory,
        DealerOrderFactory $dealerOrderFactory,
        PermHelper $permHelper,
        DealerItemRepository $dealerItemRepository
    ) {
        $this->dealerItemInterfaceFactory = $dealerItemInterfaceFactory;
        $this->_dealerOrderFactory = $dealerOrderFactory;
        $this->_permHelper = $permHelper;
        $this->dealerItemRepository = $dealerItemRepository;
    }

    public function beforePlace(
        \Magento\Sales\Model\Service\OrderService $orderService,
        OrderInterface $order
    ) {
        if ($this->_permHelper->isBackendDealer()
            && !$this->_permHelper->isAllowAllCustomersCreateOrders()
        ) { //from admin area by dealer
            $this->_permHelper->checkPermissionsByCustomerId($order->getCustomerId());
        }

        return [$order];
    }

    public function afterPlace(
        \Magento\Sales\Model\Service\OrderService $orderService,
        OrderInterface $order
    ) {
        $this->_permHelper->loadDealers($order);
        if ($this->_permHelper->hasDealers()) {
            foreach ($this->_permHelper->getDealers() as $dealer) {
                $this->assignToDealer($order, $dealer);
            }
        }

        return $order;
    }

    private function assignToDealer(OrderInterface $order, Dealer $dealer): DealerOrder
    {
        $items = $order->getItems();
        $dealerId = (int)$dealer->getId();

        foreach ($items as $product) {
            $productId = (int)$product->getId();
            $this->assignProductsToDealer($productId, $dealerId);
        }

        $dealerOrder = $this->_dealerOrderFactory->create()
            ->addData([
                'dealer_id' => $dealerId,
                'contactname' => $dealer->getContactname(),
                'order_id' => $order->getId(),
            ])
            ->save();

        return $dealerOrder;
    }

    private function assignProductsToDealer(int $productId, int $dealerId): void
    {
        $dealerItemModel = $this->dealerItemInterfaceFactory->create();
        $dealerItemModel->addData([
            'dealer_id' => $dealerId,
            'order_item_id' => $productId
        ]);
        $this->dealerItemRepository->save($dealerItemModel);
    }
}
