<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Dropshipping for Magento 2
 */

namespace Amasty\Dropshipping\Plugin;

use Amasty\Dropshipping\Helper\CollectionModifier;
use Amasty\Dropshipping\Helper\Data as PermHelper;
use Amasty\Dropshipping\Model\ResourceModel\DealerCustomer\CollectionFactory as DealerCustomerCollectionFactory;
use Amasty\Dropshipping\Model\ResourceModel\DealerOrder\CollectionFactory as DealerOrderCollectionFactory;
use Magento\Framework\App\Request\Http;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;
use Magento\Framework\View\Element\UiComponent\DataProvider\DataProvider as UiComponentDataProvider;

class DataProvider extends DataProviderCollectionFactory
{
    public const DEALERS_COLUMN = 'amasty_perm_dealers';

    /**
     * @var DealerCustomerCollectionFactory
     */
    protected $_dealerCustomerCollectionFactory;

    /**
     * @var DealerOrderCollectionFactory
     */
    protected $_dealerOrderCollectionFactory;

    /**
     * @var mixed
     */
    protected $_dealersFilter;

    /**
     * @var CollectionModifier
     */
    protected $_collectionModifier;

    /**
     * @var Http
     */
    protected $_request;

    public function __construct(
        DealerCustomerCollectionFactory $dealerCustomerCollectionFactory,
        DealerOrderCollectionFactory $dealerOrderCollectionFactory,
        CollectionModifier $collectionModifier,
        PermHelper $permHelper,
        Http $request
    ) {
        $this->_dealerCustomerCollectionFactory = $dealerCustomerCollectionFactory;
        $this->_dealerOrderCollectionFactory = $dealerOrderCollectionFactory;
        $this->_permHelper = $permHelper;
        $this->_collectionModifier = $collectionModifier;
        $this->_request = $request;
    }

    /**
     * @param UiComponentDataProvider $dataProvider
     * @param $data
     * @return mixed
     */
    public function afterGetData(
        UiComponentDataProvider $dataProvider,
        $data
    ) {
        if ($this->_collectionModifier->isCustomerDataSource($dataProvider->getName())) {
            $data = $this->_addDealersData(
                $data,
                $this->_dealerCustomerCollectionFactory,
                'entity_id',
                'customer_id'
            );
        } elseif ($this->_collectionModifier->isOrderDataSource($dataProvider->getName())) {
            $data = $this->_addDealersData(
                $data,
                $this->_dealerOrderCollectionFactory,
                'entity_id',
                'order_id'
            );
        } elseif ($this->_collectionModifier->isInvoiceDataSource($dataProvider->getName())
            || $this->_collectionModifier->isShipmentDataSource($dataProvider->getName())
            || $this->_collectionModifier->isCreditMemoDataSource($dataProvider->getName())
        ) {
            $data = $this->_addDealersData(
                $data,
                $this->_dealerOrderCollectionFactory,
                'order_id',
                'order_id'
            );
        }

        return $data;
    }

    /**
     * @param UiComponentDataProvider $dataProvider
     * @param \Closure $proceed
     * @param \Magento\Framework\Api\Filter $filter
     */
    public function aroundAddFilter(
        UiComponentDataProvider $dataProvider,
        \Closure $proceed,
        \Magento\Framework\Api\Filter $filter
    ) {
        $ret = null;

        if (($this->_collectionModifier->isCustomerDataSource($dataProvider->getName()) ||
             $this->_collectionModifier->isOrderDataSource($dataProvider->getName()) ||
             $this->_collectionModifier->isInvoiceDataSource($dataProvider->getName()) ||
             $this->_collectionModifier->isShipmentDataSource($dataProvider->getName()) ||
             $this->_collectionModifier->isCreditMemoDataSource($dataProvider->getName())
            ) && $filter->getField() === self::DEALERS_COLUMN
        ) {
            $this->_dealersFilter = $filter;
        } else {
            $ret = $proceed($filter);
        }
    }

    /**
     * @param UiComponentDataProvider $dataProvider
     * @param AbstractCollection      $collection
     *
     * @return AbstractCollection
     */
    public function afterGetSearchResult(
        \Magento\Framework\View\Element\UiComponent\DataProvider\DataProvider $dataProvider,
        $collection
    ) {
        if ($this->_collectionModifier->isOrderDataSource($dataProvider->getName())) {
            $this->_addDealersDataWithExport(
                $collection,
                $this->_dealerOrderCollectionFactory,
                'entity_id',
                'order_id'
            );
        } elseif ($this->_collectionModifier->isInvoiceDataSource($dataProvider->getName())
            || $this->_collectionModifier->isShipmentDataSource($dataProvider->getName())
            || $this->_collectionModifier->isCreditMemoDataSource($dataProvider->getName())
        ) {
            $this->_addDealersDataWithExport(
                $collection,
                $this->_dealerOrderCollectionFactory,
                'order_id',
                'order_id'
            );
        } elseif ($this->_collectionModifier->isCustomerDataSource($dataProvider->getName())) {
            $this->_addDealersDataWithExport(
                $collection,
                $this->_dealerCustomerCollectionFactory,
                'entity_id',
                'customer_id',
                true
            );
        }

        return $collection;
    }

    /**
     * @param $collection
     * @param $factory
     * @param string $primaryKey
     * @param string $foreignKey
     */
    protected function _addDealersFilter(
        $collection,
        $factory,
        $primaryKey = 'entity_id',
        $foreignKey = 'entity_id'
    ) {
        if ($this->_dealersFilter !== null) {
            $this->_collectionModifier->applyDealerFilter(
                $this->_dealersFilter->getValue(),
                $collection,
                $factory,
                $primaryKey,
                $foreignKey
            );
        }
    }

    /**
     * @param $data
     * @param $factory
     * @param string $primaryKey
     * @param string $foreignKey
     * @return mixed
     */
    protected function _addDealersData(
        $data,
        $factory,
        $primaryKey = 'entity_id',
        $foreignKey = 'entity_id'
    ) {
        if (array_key_exists('items', $data)) {
            $ids = [];
            $items = [];

            foreach ($data['items'] as &$item) {
                $ids[] = $item[$primaryKey];
                $items[$item[$primaryKey]] = &$item;
            }

            $collection = $factory->create()
                ->addDealersToSelect($ids);

            foreach ($collection as $object) {
                if (array_key_exists($object->getData($foreignKey), $items)) {
                    $item = &$items[$object->getData($foreignKey)];

                    if (!array_key_exists(self::DEALERS_COLUMN, $item)) {
                        $item[self::DEALERS_COLUMN] = [];
                    }

                    $item[self::DEALERS_COLUMN][] = $object->getContactname();
                }
            }
        }

        return $data;
    }

    /**
     * @param        $collection
     * @param        $factory
     * @param string $primaryKey
     * @param string $foreignKey
     * @param bool   $isCustomerDataSource
     *
     * @return AbstractCollection
     */
    protected function _addDealersDataWithExport(
        $collection,
        $factory,
        $primaryKey = 'entity_id',
        $foreignKey = 'entity_id',
        $isCustomerDataSource = false
    ) {
        $fullActionName = $this->_request->getFullActionName();

        if (($fullActionName == 'mui_export_gridToXml' || $fullActionName == 'mui_export_gridToCsv')
            && !$isCustomerDataSource
        ) {
            $collection = $this->_addDealersDataToExport($collection, $primaryKey, $foreignKey);
        } elseif (($fullActionName == 'mui_export_gridToXml' || $fullActionName == 'mui_export_gridToCsv')
            && $isCustomerDataSource
        ) {
            $collection = $this->_addDealersDataToCustomerExport($collection);
        } else {
            $this->_addDealersFilter($collection, $factory, $primaryKey, $foreignKey);
        }

        return $collection;
    }

    /**
     * @param AbstractCollection $collection
     * @param string             $primaryKey
     * @param string             $foreignKey
     *
     * @return AbstractCollection
     */
    protected function _addDealersDataToExport(
        AbstractCollection $collection,
        $primaryKey = 'entity_id',
        $foreignKey = 'entity_id'
    ) {
        $connection = $collection->getConnection();
        $collection->getSelect()->joinLeft(
            ['amdealerorder' => $collection->getTable('amasty_perm_dealer_order')],
            $connection->quoteIdentifier('main_table.' . $primaryKey) .
                ' = '. $connection->quoteIdentifier('amdealerorder.' . $foreignKey),
            ['main_table.*', self::DEALERS_COLUMN => 'GROUP_CONCAT(amdealerorder.contactname)']
        );
        $collection->getSelect()->group('main_table.entity_id');
        $collection->load();

        return $collection;
    }

    /**
     * @param AbstractCollection $collection
     *
     * @return AbstractCollection
     */
    protected function _addDealersDataToCustomerExport($collection)
    {
        $collection->getSelect()->joinLeft(
            ['adc' => $collection->getTable('amasty_perm_dealer_customer')],
            'main_table.entity_id = adc.`customer_id`'
        )->joinLeft(
            ['ad' => $collection->getTable('amasty_perm_dealer')],
            'ad.entity_id = adc.dealer_id'
        )->joinLeft(
            ['au' => $collection->getTable('admin_user')],
            'au.user_id = ad.user_id',
            ['main_table.*', 'CONCAT_WS(" ", au.firstname, au.lastname) AS ' . self::DEALERS_COLUMN]
        );
        $collection->load();

        return $collection;
    }
}
