<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Dropshipping for Magento 2
 */

namespace Amasty\Dropshipping\Model;

use Magento\Framework\App\Area;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Payment\Helper\Data;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Model\Order\Address\Renderer;

class Mailer
{
    /**
     * @var TransportBuilder
     */
    protected $transportBuilder;

    /**
     * @var string
     */
    protected $emailTemplate;

    /**
     * @var Renderer
     */
    private $addressRenderer;

    /**
     * @var Data
     */
    private $paymentHelper;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        TransportBuilder $transportBuilder,
        Renderer $addressRenderer,
        Data $paymentHelper,
        ConfigProvider $configProvider
    ) {
        $this->transportBuilder = $transportBuilder;
        $this->addressRenderer = $addressRenderer;
        $this->paymentHelper = $paymentHelper;
        $this->configProvider = $configProvider;
    }

    /**
     * @param string         $email
     * @param OrderInterface $order
     * @param array          $vars
     *
     * @return void
     * @throws LocalizedException
     * @throws MailException
     * @throws NoSuchEntityException
     */
    public function send($email, $order, $vars = []): void
    {
        $storeId = (int)$order->getStoreId();
        $this->initTemplate($vars, $storeId);

        if ($this->configProvider->isNotifyEnabled($storeId)) {
            $billingAddress = $this->addressRenderer->format($order->getBillingAddress(), 'html');
            $shippingAddress = $this->addressRenderer->format($order->getShippingAddress(), 'html');
            $adminName = $this->configProvider->getAdminName($storeId);

            $templateVars = array_merge(
                [
                    'order_customer_name' => $order->getCustomerName(),
                    'order_status_label' => $order->getStatusLabel(),
                    'order_is_not_virtual' => $order->getIsNotVirtual(),
                    'order_shipping_description' => $order->getShippingDescription(),
                    'incrementId'              => $order->getIncrementId(),
                    'formattedBillingAddress'  => $billingAddress,
                    'formattedShippingAddress' => $shippingAddress,
                    'order_id'                 => $order->getId(),
                    'paymentHtml'              => $this->getPaymentHtml($order),
                    'adminName'                => $adminName
                ],
                $vars
            );

            $this->transportBuilder
                ->setTemplateIdentifier($this->emailTemplate)
                ->setTemplateOptions(
                    [
                        'area'  => Area::AREA_FRONTEND,
                        'store' => $storeId
                    ]
                )->setTemplateVars(
                    $templateVars
                )->setFromByScope(
                    $this->configProvider->getEmailsSenderIdentity($storeId),
                    $storeId
                )->addTo(
                    $email
                );

            if ($copyToEmails = $this->configProvider->getEmailCopyTo($storeId)) {
                $this->transportBuilder->addTo($copyToEmails);
            }
            $transport = $this->transportBuilder->getTransport();

            $transport->sendMessage();
        }
    }

    /**
     * Returns payment info block as HTML.
     *
     * @param OrderInterface $order
     *
     * @return string
     * @throws \Exception
     */
    private function getPaymentHtml(OrderInterface $order)
    {
        return $this->paymentHelper->getInfoBlockHtml(
            $order->getPayment(),
            $order->getStoreId()
        );
    }

    private function initTemplate(array $vars, int $storeId): void
    {
        if (empty($vars['comment'])) {
            $this->emailTemplate = $this->configProvider->getAssignTemplateId($storeId);
        } else {
            $this->emailTemplate = $this->configProvider->getCommentTemplateId($storeId);
        }
    }
}
