<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Dropshipping for Magento 2
 */

namespace Amasty\Dropshipping\Model\DealerItem;

use Amasty\Dropshipping\Api\Data\DealerItemInterface;
use Amasty\Dropshipping\Api\Data\DealerItemInterfaceFactory;
use Amasty\Dropshipping\Api\DealerItemRepositoryInterface;
use Amasty\Dropshipping\Model\Dealer;
use Amasty\Dropshipping\Model\DealerItem\ResourceModel\Collection;
use Amasty\Dropshipping\Model\DealerItem\ResourceModel\CollectionFactory;
use Amasty\Dropshipping\Model\DealerItem\ResourceModel\DealerItem as DealerItemResource;
use Amasty\Dropshipping\Model\DealerOrderFactory;
use Magento\Framework\Api\Search\FilterGroup;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SortOrder;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Ui\Api\Data\BookmarkSearchResultsInterfaceFactory;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class DealerItemRepository implements DealerItemRepositoryInterface
{
    /**
     * @var BookmarkSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @var DealerItemFactory
     */
    private $dealerItemFactory;

    /**
     * @var DealerItemResource
     */
    private $dealerItemResource;

    /**
     * Model data storage
     *
     * @var array
     */
    private $dealerItems;

    /**
     * @var CollectionFactory
     */
    private $dealerItemCollectionFactory;

    /**
     * @var DealerOrderFactory
     */
    private $dealerOrderFactory;

    /**
     * @var DealerItemInterfaceFactory
     */
    private $dealerItemInterfaceFactory;

    /**
     * @var Dealer
     */
    private $dealer;

    public function __construct(
        BookmarkSearchResultsInterfaceFactory $searchResultsFactory,
        DealerItemFactory $dealerItemFactory,
        DealerItemResource $dealerItemResource,
        CollectionFactory $dealerItemCollectionFactory,
        DealerOrderFactory $dealerOrderFactory,
        DealerItemInterfaceFactory $dealerItemInterfaceFactory,
        Dealer $dealer
    ) {
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dealerItemFactory = $dealerItemFactory;
        $this->dealerItemResource = $dealerItemResource;
        $this->dealerItemCollectionFactory = $dealerItemCollectionFactory;
        $this->dealerOrderFactory = $dealerOrderFactory;
        $this->dealerItemInterfaceFactory = $dealerItemInterfaceFactory;
        $this->dealer = $dealer;
    }

    public function save(DealerItemInterface $dealerItem)
    {
        try {
            if ($dealerItemModel = $this->isNewItem($dealerItem)) {
                $dealerItemModel->setDealerId($dealerItem->getDealerId());
                $dealerItemModel->setNotifyStatus($dealerItem->getNotifyStatus());
            } else {
                $dealerItemModel = $this->dealerItemFactory->create()->addData($dealerItem->getData());
            }
            $this->dealerItemResource->save($dealerItemModel);
            unset($this->dealerItems[$dealerItemModel->getEntityId()]);
        } catch (\Exception $e) {
            if ($dealerItem->getEntityId()) {
                throw new CouldNotSaveException(
                    __(
                        'Unable to save dealerItem with ID %1. Error: %2',
                        [$dealerItem->getEntityId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotSaveException(__('Unable to save new dealerItem. Error: %1', $e->getMessage()));
        }

        return $dealerItem;
    }

    public function getById($entityId)
    {
        if (!isset($this->dealerItems[$entityId])) {
            /** @var \Amasty\Dropshipping\Model\DealerItem\DealerItem $dealerItem */
            $dealerItem = $this->dealerItemFactory->create();
            $this->dealerItemResource->load($dealerItem, $entityId);
            if (!$dealerItem->getEntityId()) {
                throw new NoSuchEntityException(__('DealerItem with specified ID "%1" not found.', $entityId));
            }
            $this->dealerItems[$entityId] = $dealerItem;
        }

        return $this->dealerItems[$entityId];
    }

    /**
     * @param $orderItemId
     *
     * @return \Magento\Framework\DataObject
     */
    public function getByOrderItemId($orderItemId)
    {
        return $this->dealerItemCollectionFactory->create()
            ->addFieldToFilter(DealerItem::ORDER_ITEM_ID, $orderItemId)
            ->getFirstItem();
    }

    public function isNewItem($currentDealerItem)
    {
        $orderItemId = $currentDealerItem->getOrderItemId();
        /** @var \Amasty\Dropshipping\Model\DealerItem\DealerItem $currentDealerItem */
        if (!isset($this->dealerItems[$orderItemId])) {
            $dealerItem = $this->dealerItemFactory->create();
            $this->dealerItemResource->load($dealerItem, $orderItemId, DealerItemInterface::ORDER_ITEM_ID);
            if ($dealerItem->getEntityId()) {
                $this->dealerItems[$orderItemId] = $dealerItem;
            } else {
                return $currentDealerItem;
            }
        }

        return $this->dealerItems[$orderItemId];
    }

    public function delete(DealerItemInterface $dealerItem)
    {
        try {
            $this->dealerItemResource->delete($dealerItem);
            unset($this->dealerItems[$dealerItem->getEntityId()]);
        } catch (\Exception $e) {
            if ($dealerItem->getEntityId()) {
                throw new CouldNotDeleteException(
                    __(
                        'Unable to remove dealerItem with ID %1. Error: %2',
                        [$dealerItem->getEntityId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotDeleteException(__('Unable to remove dealerItem. Error: %1', $e->getMessage()));
        }

        return true;
    }

    public function deleteById($entityId)
    {
        $dealerItemModel = $this->getById($entityId);
        $this->delete($dealerItemModel);

        return true;
    }

    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);

        /** @var \Amasty\Dropshipping\Model\ResourceModel\DealerItem\Collection $dealerItemCollection */
        $dealerItemCollection = $this->dealerItemCollectionFactory->create();

        // Add filters from root filter group to the collection
        foreach ($searchCriteria->getFilterGroups() as $group) {
            $this->addFilterGroupToCollection($group, $dealerItemCollection);
        }

        $searchResults->setTotalCount($dealerItemCollection->getSize());
        $sortOrders = $searchCriteria->getSortOrders();

        if ($sortOrders) {
            $this->addOrderToCollection($sortOrders, $dealerItemCollection);
        }

        $dealerItemCollection->setCurPage($searchCriteria->getCurrentPage());
        $dealerItemCollection->setPageSize($searchCriteria->getPageSize());

        $dealerItems = [];
        /** @var DealerItemInterface $dealerItem */
        foreach ($dealerItemCollection->getItems() as $dealerItem) {
            $dealerItems[] = $this->getById($dealerItem->getEntityId());
        }

        $searchResults->setItems($dealerItems);

        return $searchResults;
    }

    /**
     * Helper function that adds a FilterGroup to the collection.
     *
     * @param FilterGroup $filterGroup
     * @param Collection  $dealerItemCollection
     *
     * @return void
     */
    private function addFilterGroupToCollection(FilterGroup $filterGroup, Collection $dealerItemCollection)
    {
        foreach ($filterGroup->getFilters() as $filter) {
            $condition = $filter->getConditionType() ?: 'eq';
            $dealerItemCollection->addFieldToFilter($filter->getField(), [$condition => $filter->getValue()]);
        }
    }

    /**
     * Helper function that adds a SortOrder to the collection.
     *
     * @param SortOrder[] $sortOrders
     * @param Collection  $dealerItemCollection
     *
     * @return void
     */
    private function addOrderToCollection($sortOrders, Collection $dealerItemCollection)
    {
        /** @var SortOrder $sortOrder */
        foreach ($sortOrders as $sortOrder) {
            $field = $sortOrder->getField();
            $dealerItemCollection->addOrder(
                $field,
                ($sortOrder->getDirection() == SortOrder::SORT_DESC) ? SortOrder::SORT_DESC : SortOrder::SORT_ASC
            );
        }
    }

    /**
     * @param array $item
     */
    public function saveDealerOrder($item)
    {
        if (!isset($item['contactname']) && isset($item['dealer_id'])) {
            $item['contactname'] = $this->dealer->getContactname($item['dealer_id']);
        }

        $dealerOrder = $this->dealerOrderFactory->create();
        $dealerOrder->addData($item);
        $dealerOrder->save();
    }

    /**
     * @param $item
     * @param $dealerId
     */
    public function processOrderItems($item, $dealerId)
    {
        if ($childItems = $item->getChildrenItems()) {
            foreach ($childItems as $childItem) {
                $this->saveDealerOrderItems($childItem['item_id'], $dealerId);
            }
        }

        $this->saveDealerOrderItems($item->getId(), $dealerId);
    }

    /**
     * @param int $itemId
     * @param int $dealerId
     */
    private function saveDealerOrderItems($itemId, $dealerId)
    {
        $dealerItemModel = $this->dealerItemInterfaceFactory->create();
        $dealerItemModel->addData(['dealer_id' => (int)$dealerId, 'order_item_id' => (int)$itemId]);
        $this->save($dealerItemModel);
    }
}
