<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Dropshipping for Magento 2
 */
namespace Amasty\Dropshipping\Helper;

use Amasty\Dropshipping\Model\Dealer;
use Amasty\Dropshipping\Model\ResourceModel\Dealer\CollectionFactory as DealerCollectionFactory;
use Amasty\Dropshipping\Model\ResourceModel\DealerCustomer\CollectionFactory as DealerCustomerCollectionFactory;
use Amasty\Dropshipping\Model\ResourceModel\DealerOrder\CollectionFactory as DealerOrderCollectionFactory;
use Magento\Sales\Model\Order;

class Data extends \Magento\Framework\App\Helper\AbstractHelper
{
    public const SCOPE_GENERAL_SINGLE_DEALER = 'amasty_dropshipping/general/single_dealer';
    public const SCOPE_GENERAL_DEFAULT_DEALER = 'amasty_dropshipping/general/default_dealer';
    public const SCOPE_GENERAL_REASSIGN_FIELDS = 'amasty_dropshipping/general/reassign_fields';
    public const SCOPE_GENERAL_FROM_TO = 'amasty_dropshipping/general/from_to';
    public const SCOPE_GENERAL_AUTHOR = 'amasty_dropshipping/general/author';
    public const SCOPE_GENERAL_EDIT_NO_GRID = 'amasty_dropshipping/general/edit_no_grid';
    public const SCOPE_GENERAL_ALLOW_ALL_CUSTOMERS_AND_ORDERS =
        'amasty_dropshipping/general/allow_all_customers_and_orders';
    public const SCOPE_GENERAL_ALLOW_ALL_CUSTOMERS_CREATE_ORDERS =
        'amasty_dropshipping/general/allow_all_customers_create_orders';

    public const SCOPE_FRONTEND_ON_REGISTRATION = 'amasty_dropshipping/frontend/on_registration';
    public const SCOPE_FRONTEND_IN_ACCOUNT = 'amasty_dropshipping/frontend/in_account';
    public const SCOPE_FRONTEND_DESCRIPTION_CHECKOUT = 'amasty_dropshipping/frontend/description_checkout';

    public const FROM_USER_EDIT = 'from_user_edit';

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $_scopeConfig;

    /**
     * @var bool
     */
    protected $_singleDealerMode;

    /**
     * @var bool
     */
    protected $_sendEmailMode;

    /**
     * @var bool
     */
    protected $_reassignFieldsMode;

    /**
     * @var bool
     */
    protected $_fromToMode;

    /**
     * @var bool
     */
    protected $_authorMode;

    /**
     * @var bool
     */
    protected $_editNoGridMode;

    /**
     * @var bool
     */
    protected $_isOnRegistrationMode;

    /**
     * @var bool
     */
    protected $_isInAccountMode;

    /**
     * @var bool
     */
    protected $_isDescriptionCheckout;

    /**
     * @var \Magento\Backend\Model\Auth\Session
     */
    protected $_authSession;

    /**
     * @var Dealer
     */
    protected $_backendDealer;

    /**
     * @var array
     */
    protected $_dealers;

    /**
     * @var \Amasty\Dropshipping\Model\DealerFactory
     */
    protected $_dealerFactory;

    /**
     * @var DealerCustomerCollectionFactory
     */
    protected $_dealerCustomerCollectionFactory;

    /**
     * @var DealerOrderCollectionFactory
     */
    protected $_dealerOrderCollectionFactory;

    /**
     * @var DealerCollectionFactory
     */
    protected $_dealerCollectionFactory;

    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Backend\Model\Auth\Session $authSession,
        \Amasty\Dropshipping\Model\DealerFactory $dealerFactory,
        DealerCustomerCollectionFactory $dealerCustomerCollectionFactory,
        DealerOrderCollectionFactory $dealerOrderCollectionFactory,
        DealerCollectionFactory $dealerCollectionFactory
    ) {
        parent::__construct($context);
        $this->_scopeConfig = $context->getScopeConfig();
        $this->_authSession = $authSession;
        $this->_dealerFactory = $dealerFactory;
        $this->_dealerCustomerCollectionFactory = $dealerCustomerCollectionFactory;
        $this->_dealerOrderCollectionFactory = $dealerOrderCollectionFactory;
        $this->_dealerCollectionFactory = $dealerCollectionFactory;
    }

    public function getScopeValue($path)
    {
        return $this->_scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
    }

    public function isSingleDealerMode()
    {
        if ($this->_singleDealerMode === null) {
            $this->_singleDealerMode = $this->getScopeValue(self::SCOPE_GENERAL_SINGLE_DEALER) === '1';
        }
        return $this->_singleDealerMode;
    }

    public function isReassignFieldsMode()
    {
        if ($this->_reassignFieldsMode === null) {
            $this->_reassignFieldsMode = $this->getScopeValue(self::SCOPE_GENERAL_REASSIGN_FIELDS) === '1';
        }
        return $this->_reassignFieldsMode;
    }

    public function isFromToMode()
    {
        if ($this->_fromToMode === null) {
            $this->_fromToMode = $this->getScopeValue(self::SCOPE_GENERAL_FROM_TO) === '1';
        }
        return $this->_fromToMode;
    }

    public function isAuthorMode()
    {
        if ($this->_authorMode === null) {
            $this->_authorMode = $this->getScopeValue(self::SCOPE_GENERAL_AUTHOR) === '1';
        }
        return $this->_authorMode;
    }

    public function isEditNoGridMode()
    {
        if ($this->_editNoGridMode === null) {
            $this->_editNoGridMode = $this->getScopeValue(self::SCOPE_GENERAL_EDIT_NO_GRID) === '1';
        }
        return $this->_editNoGridMode;
    }

    public function isOnRegistrationMode()
    {
        if ($this->_isOnRegistrationMode === null) {
            $this->_isOnRegistrationMode = $this->getScopeValue(self::SCOPE_FRONTEND_ON_REGISTRATION) === '1';
        }
        return $this->_isOnRegistrationMode;
    }

    public function isInAccountMode()
    {
        if ($this->_isInAccountMode === null) {
            $this->_isInAccountMode = $this->getScopeValue(self::SCOPE_FRONTEND_IN_ACCOUNT) === '1';
        }
        return $this->_isInAccountMode;
    }

    public function isDescriptionCheckoutMode()
    {
        if ($this->_isDescriptionCheckout === null) {
            $this->_isDescriptionCheckout = $this->getScopeValue(self::SCOPE_FRONTEND_DESCRIPTION_CHECKOUT) === '1';
        }
        return $this->_isDescriptionCheckout;
    }

    public function isBackendDealer()
    {
        return $this->getBackendDealer() !== null
            && $this->getBackendDealer()->getId() !== null
            && $this->getBackendDealer()->checkPermissions();
    }

    public function isAllowAllCustomersAndOrders()
    {
        return $this->getScopeValue(self::SCOPE_GENERAL_ALLOW_ALL_CUSTOMERS_AND_ORDERS);
    }

    public function isAllowAllCustomersCreateOrders()
    {
        return $this->getScopeValue(self::SCOPE_GENERAL_ALLOW_ALL_CUSTOMERS_CREATE_ORDERS);
    }

    /**
     * @return Dealer
     */
    public function getBackendDealer()
    {
        if ($this->_backendDealer === null) {
            $user = $this->_authSession->getUser();

            if ($user) {
                $this->_backendDealer = $this->_dealerFactory->create()
                    ->load($user->getId(), 'user_id');

                if (!$this->_backendDealer->getRoleId()) {
                    $this->_backendDealer->setId(null);
                }
            }
        }

        return $this->_backendDealer;
    }

    /**
     * @return int
     */
    public function getBackendDealerId()
    {
        if (is_object($this->_backendDealer)) {
            return $this->_backendDealer->getId();
        }
    }

    public function hasDealers()
    {
        $dealers = $this->getDealers();
        return count($dealers) > 0;
    }

    public function loadDealers(Order $order)
    {
        if ($this->_dealers === null) {
            $this->_dealers = [];

            if ($this->isBackendDealer()) {
                $this->_fillBackendDealers();
            } elseif ($order->getCustomerId()) {
                $this->_fillFrontendDealers($order->getCustomerId());
            }

            if (count($this->_dealers) === 0) {
                $this->_fillDefaultDealers();
            }
        }

        return $this->_dealers;
    }

    public function getDealers()
    {
        return $this->_dealers !== null ? $this->_dealers : [];
    }

    protected function _fillDefaultDealers()
    {
        $dealerId = $this->getScopeValue(self::SCOPE_GENERAL_DEFAULT_DEALER);

        if ($dealerId > 0) {
            $dealer = $this->_dealerFactory->create()
                ->load($dealerId);

            if ($dealer->checkPermissions()) {
                $this->_dealers[] = $dealer;
            }
        }
    }

    protected function _fillBackendDealers()
    {
        $dealer = $this->getBackendDealer();

        if ($dealer->checkPermissions()) {
            $this->_dealers[] = $dealer;
        }
    }

    protected function _fillFrontendDealers($customerId)
    {
        $dealerCustomerCollection = $this->_dealerCustomerCollectionFactory->create()
            ->addFieldToFilter('customer_id', $customerId);

        $dealersIds = $dealerCustomerCollection->getDealersIds();

        if (count($dealersIds) > 0) {
            $dealerCollection = $this->_dealerCollectionFactory->create()
                ->addUserData()
                ->addFieldToFilter('main_table.entity_id', ['in' => $dealersIds]);

            foreach ($dealerCollection as $dealer) {
                if ($dealer->checkPermissions()) {
                    $this->_dealers[] = $dealer;
                }
            }
        }
    }

    protected function _checkPermiossionsByDealersIds(array $dealersIds)
    {
        if (!in_array($this->getBackendDealer()->getId(), $dealersIds)) {
            throw new \Magento\Framework\Exception\NoSuchEntityException(
                __(
                    "%1 don't have permissions for order",
                    $this->getBackendDealer()->getContactname()
                )
            );
        }
    }

    public function checkPermissionsByOrder(Order $order)
    {
        if ($this->isAllowAllCustomersAndOrders()) {
            return $order;
        }

        if ($this->isBackendDealer()) {
            $dealerOrderCollection = $this->_dealerOrderCollectionFactory->create()
                ->addFieldToFilter('order_id', $order->getId());

            $dealersIds = $dealerOrderCollection->getDealersIds();

            $this->_checkPermiossionsByDealersIds($dealersIds);
        }
    }

    public function checkPermissionsByCustomerId($customerId)
    {
        if ($this->isBackendDealer()) {
            $dealerCustomerCollection = $this->_dealerCustomerCollectionFactory->create()
                ->addFieldToFilter('customer_id', $customerId);

            $dealersIds = $dealerCustomerCollection->getDealersIds();

            $this->_checkPermiossionsByDealersIds($dealersIds);
        }
    }
}
