<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Dropshipping for Magento 2
 */

namespace Amasty\Dropshipping\Controller\Adminhtml\Order;

use Amasty\Dropshipping\Model\DealerFactory;
use Amasty\Dropshipping\Model\DealerItem\DealerItemRepository;
use Amasty\Dropshipping\Model\DealerItem\Notified\Statuses;
use Amasty\Dropshipping\Model\DealerItem\ResourceModel\DealerItem;
use Amasty\Dropshipping\Model\DealerItem\ResourceModel\DealerItem as DealerItemResource;
use Amasty\Dropshipping\Model\Mailer;
use Amasty\Dropshipping\Model\ResourceModel\DealerOrder as DealerOrderResource;
use Magento\Backend\App\Action;
use Magento\Framework\Controller\Result\Json;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;

class AssignDealer extends \Amasty\Dropshipping\Controller\Adminhtml\Massaction
{
    /**
     * @var DealerOrderResource
     */
    private $dealerOrderResource;

    /**
     * @var DealerItemRepository
     */
    private $dealerItemRepository;

    /**
     * @var DealerItemResource
     */
    protected $dealerItemResource;

    /**
     * @var CollectionFactory
     */
    private $orderCollectionFactory;

    public function __construct(
        Action\Context $context,
        DealerFactory $dealerFactory,
        Mailer $mailer,
        CollectionFactory $orderCollectionFactory,
        DealerOrderResource $dealerOrderResource,
        DealerItemResource $dealerItemResource,
        DealerItemRepository $dealerItemRepository,
        OrderRepositoryInterface $orderRepository
    ) {
        $this->dealerFactory = $dealerFactory;
        $this->mailer = $mailer;
        $this->dealerOrderResource = $dealerOrderResource;
        $this->dealerItemRepository = $dealerItemRepository;
        $this->dealerItemResource = $dealerItemResource;
        parent::__construct($context, $dealerFactory, $orderRepository, $mailer);
        $this->orderCollectionFactory = $orderCollectionFactory;
    }

    public function execute()
    {
        $status = Statuses::NOTIFIED;
        $result = [];
        $alreadySaved = [];
        /** @var Json $response */
        $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        if ($data = $this->getRequest()->getParams()) {
            if (isset($data['dealersData'])) {
                $orderId = (int)$data['order_id'];
                $this->dealerOrderResource->cleanByOrderId($orderId);
                $order = $this->orderCollectionFactory->create()
                    ->addFieldToFilter('entity_id', $orderId)
                    ->getFirstItem();
                foreach ($data['dealersData'] as $orderItemId => $dealerId) {
                    if ($dealerId == 0) {
                        $this->dealerItemResource->cleanByOrderItemId($orderItemId);
                        $result[Statuses::NOT_NOTIFIED][] = $orderItemId;
                        continue;
                    }
                    $orderItem = $order->getItemById($orderItemId);
                    $this->dealerItemRepository->processOrderItems($orderItem, $dealerId);

                    if (!isset($alreadySaved[$dealerId]) || $alreadySaved[$dealerId] != $orderId) {
                        $alreadySaved[$dealerId] = $orderId;
                        $dealerItemMapping = $this->getDealerItemMapping($data['dealersData']);
                        $item['order_id'] = $orderId;
                        $item['dealer_id'] = $dealerId;
                        $this->dealerItemRepository->saveDealerOrder($item);
                        try {
                            $orderItemIds = $dealerItemMapping[$dealerId];
                            $this->sendNotifications($orderItemIds, $dealerId, $orderId);
                            $status = Statuses::NOTIFIED;
                        } catch (\Exception $e) {
                            $this->messageManager->addErrorMessage($e->getMessage());
                            $status = Statuses::FAILED;
                        }
                        foreach ($orderItemIds as $itemId) {
                            $result[$status][] = $itemId;
                        }
                    }
                }
            }
        }

        $this->processResult($result);

        return $response->setData($result);
    }

    /**
     * @param $data
     *
     * @return array
     */
    private function getDealerItemMapping($data)
    {
        $dealerItemMapping = [];
        foreach ($data as $orderItemId => $dealerId) {
            $dealerItemMapping[$dealerId][] = $orderItemId;
        }

        return $dealerItemMapping;
    }

    /**
     * @param array $result
     */
    private function processResult($result)
    {
        try {
            foreach ($result as $status => $id) {
                $this->dealerItemResource->updateNotifyStatus(
                    array_values($id),
                    $status
                );
            }
        } catch (LocalizedException $e) {
            $this->messageManager->addErrorMessage($e->getMessage());
        }
    }
}
