<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Dropshipping for Magento 2
 */

namespace Amasty\Dropshipping\Controller\Adminhtml\Order;

use Amasty\Dropshipping\Model\ConfigProvider;
use Amasty\Dropshipping\Model\DealerOrder;
use Amasty\Dropshipping\Model\DealerOrder\AssignHistory;
use Amasty\Dropshipping\Model\Mailer;
use Amasty\Dropshipping\Helper\Data;
use Magento\Backend\App\Action;
use Magento\Framework\Controller\ResultFactory;
use Magento\Sales\Api\OrderManagementInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Psr\Log\LoggerInterface;

class AddDealerComment extends \Magento\Sales\Controller\Adminhtml\Order
{
    /**
     * @var DealerOrder
     */
    protected $dealerOrder;

    /**
     * @var Mailer
     */
    private $mailer;

    /**
     * @var Data
     */
    private $helper;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        Action\Context $context,
        \Magento\Framework\Registry $coreRegistry,
        \Magento\Framework\App\Response\Http\FileFactory $fileFactory,
        \Magento\Framework\Translate\InlineInterface $translateInline,
        \Magento\Framework\View\Result\PageFactory $resultPageFactory,
        \Magento\Framework\Controller\Result\JsonFactory $resultJsonFactory,
        \Magento\Framework\View\Result\LayoutFactory $resultLayoutFactory,
        \Magento\Framework\Controller\Result\RawFactory $resultRawFactory,
        OrderManagementInterface $orderManagement,
        OrderRepositoryInterface $orderRepository,
        LoggerInterface $logger,
        DealerOrder $dealerOrder,
        Mailer $mailer,
        Data $helper,
        ConfigProvider $configProvider
    ) {
        parent::__construct(
            $context,
            $coreRegistry,
            $fileFactory,
            $translateInline,
            $resultPageFactory,
            $resultJsonFactory,
            $resultLayoutFactory,
            $resultRawFactory,
            $orderManagement,
            $orderRepository,
            $logger
        );
        $this->dealerOrder = $dealerOrder;
        $this->mailer = $mailer;
        $this->helper = $helper;
        $this->configProvider = $configProvider;
    }

    /**
     * @param $orderId
     * @return mixed
     */
    protected function _initDealerOrder($orderId)
    {
        if ($this->dealerOrder === null) {
            $this->dealerOrder = $this->dealerOrder->load($orderId, 'order_id');
            if (!$this->dealerOrder->getId()) {
                $this->dealerOrder->setOrderId($orderId);
            }
        }
        return $this->dealerOrder;
    }

    /**
     * @return $this|\Magento\Framework\Controller\Result\Json|\Magento\Framework\View\Result\Page
     */
    public function execute()
    {
        $toDealer = false;
        $order = $this->_initOrder();
        if ($order) {
            try {
                $data = $this->getRequest()->getParams();

                /** @var  DealerOrder $dealerOrder */
                $dealerOrder = $this->_initDealerOrder($order->getId());

                if (isset($data['dealer_id']) && $data['dealer_id']) {
                    $toDealer = (int)$data['dealer_id'];
                    $receiverEmail = $dealerOrder->getDealer(
                        (int)$order->getStoreId(),
                        true
                    )->getDealerEmail($toDealer);
                } else {
                    $receiverEmail = $this->configProvider->getAdminEmails((int)$order->getStoreId());
                }

                if (empty($data['comment'])) {
                    throw new \Magento\Framework\Exception\LocalizedException(__('Please enter a comment.'));
                }

                if (empty($receiverEmail)) {
                    throw new \Magento\Framework\Exception\LocalizedException(__('User doesn\'t have email.'));
                }

                $notify = isset($data['is_dealer_notified']) ? $data['is_dealer_notified'] : false;

                /** @var AssignHistory $history */
                $history = $dealerOrder->addDealerHistoryComment($data['comment'], $order->getId(), $toDealer);

                $history->setIsDealerNotified($notify);
                $history->save();

                if ($notify) {
                    $comment = trim(strip_tags($data['comment']));
                    $vars = [
                        'incrementId' => $order->getIncrementId(),
                        'comment' => $comment,
                        'billing' => $order->getBillingAddress(),
                        'store' => $order->getStore(),
                        'history' => $history,
                        'dealerOrder' => $dealerOrder,
                        'from_dealer_contactname' => $history->getFromDealerContactname(),
                        'to_dealer_contactname' => $history->getToDealerContactname(),
                        'author_dealer_contactname' => $history->getAuthorDealerContactname()
                    ];

                    $this->mailer->send($receiverEmail, $order, $vars);
                }

                $resultLayout = $this->resultFactory->create(ResultFactory::TYPE_LAYOUT);

                $resultLayout->getLayout()->getBlock('amasty.perm.assign');

                return $resultLayout;
            } catch (\Magento\Framework\Exception\LocalizedException $e) {
                $response = ['error' => true, 'message' => $e->getMessage()];
            } catch (\Exception $e) {
                $response = ['error' => true, 'message' => $e->getMessage() . __('We cannot assign dealer.')];
            }
            if (is_array($response)) {
                $resultJson = $this->resultJsonFactory->create();
                $resultJson->setData($response);
                return $resultJson;
            }
        }
        return $this->resultRedirectFactory->create()->setPath('sales/*/');
    }
}
