<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Dropshipping for Magento 2
 */

namespace Amasty\Dropshipping\Controller\Adminhtml\Massaction;

use Amasty\Dropshipping\Model\DealerFactory;
use Amasty\Dropshipping\Model\DealerItem\DealerItemRepository;
use Amasty\Dropshipping\Model\DealerItem\Notified\Statuses;
use Amasty\Dropshipping\Model\DealerItem\ResourceModel\DealerItem;
use Amasty\Dropshipping\Model\Mailer;
use Amasty\Dropshipping\Model\ResourceModel\DealerOrder as DealerOrderResource;
use Magento\Backend\App\Action;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;
use Magento\Ui\Component\MassAction\Filter;

class Index extends \Amasty\Dropshipping\Controller\Adminhtml\Massaction
{
    /**
     * @var DealerItemRepository
     */
    private $dealerItemRepository;

    /**
     * @var Filter
     */
    private $filter;

    /**
     * @var DealerOrderResource
     */
    private $dealerOrderResource;

    /**
     * @var CollectionFactory
     */
    private $orderCollectionFactory;

    /**
     * @var DealerItem
     */
    private $dealerItemResource;

    public function __construct(
        Action\Context $context,
        Filter $filter,
        CollectionFactory $orderCollectionFactory,
        DealerItem $dealerItemResource,
        DealerItemRepository $dealerItemRepository,
        DealerFactory $dealerFactory,
        DealerOrderResource $dealerOrderResource,
        Mailer $mailer,
        OrderRepositoryInterface $orderRepository
    ) {
        $this->filter = $filter;
        $this->orderCollectionFactory = $orderCollectionFactory;
        $this->dealerFactory = $dealerFactory;
        $this->dealerOrderResource = $dealerOrderResource;
        $this->mailer = $mailer;
        $this->dealerItemRepository = $dealerItemRepository;
        parent::__construct($context, $dealerFactory, $orderRepository, $mailer);
        $this->dealerItemResource = $dealerItemResource;
    }

    /**
     * Assign dealer to orders
     *
     * @return \Magento\Framework\App\ResponseInterface
     *
     * @throws LocalizedException
     */
    public function execute()
    {
        $status = Statuses::NOTIFIED;
        $orderIncrementIds = [];
        $dealerId = (int)$this->getRequest()->getParam('dealer');
        $dealer = $this->dealerFactory->create()->load($dealerId);
        $dealer->checkPermissions(); // added contactname to dealer
        $contactName = $dealer->getContactname();

        $this->filter->applySelectionOnTargetProvider();
        $collection = $this->filter->getCollection($this->orderCollectionFactory->create());
        $ids = $collection->getAllIds();

        try {
            foreach ($ids as $id) {
                $order = $this->orderCollectionFactory->create()
                    ->addFieldToFilter('entity_id', $id)
                    ->getFirstItem();
                $orderIncrementIds[] = $order->getIncrementId();
                $this->dealerOrderResource->cleanByOrderId($order->getId());
                $item = [
                    'dealer_id' => $dealerId,
                    'order_id' => $order->getId(),
                    'contactname' => $contactName
                ];
                $this->dealerItemRepository->saveDealerOrder($item);
                $orderItems = $order->getAllItems();
                foreach ($orderItems as $item) {
                    $this->dealerItemRepository->processOrderItems($item, $dealerId);
                }
                $this->mailer->send(
                    $dealer->getEmail(),
                    $order
                );
            }
            $this->messageManager->addSuccessMessage(__('Dealer was successfully assigned.'));
        } catch (\Exception $e) {
            $this->messageManager->addErrorMessage($e->getMessage());
            $status = Statuses::FAILED;
        }

        $this->dealerItemResource->updateNotifyStatus(
            array_keys($order->getItems()),
            $status
        );

        /** @var \Magento\Backend\Model\View\Result\Redirect $resultRedirect */
        $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);

        return $resultRedirect->setPath('sales/order/index');
    }
}
