<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/

namespace Amasty\Segments\Traits;

use Magento\Framework\Model\AbstractModel;

/**
 * Used in Conditions.
 * Add Input type "day"
 */
trait DayValidation
{
    /**
     * Return real Order attribute for validate
     *
     * @return string
     */
    protected function getEavAttributeCode(): string
    {
        return $this->getAttribute();
    }

    /**
     * Default operator input by type map getter
     *
     * @return array
     */
    public function getDefaultOperatorInputByType(): array
    {
        $operators = parent::getDefaultOperatorInputByType();
        $operators['day'] = ['==', '>=', '>', '<=', '<'];

        return $operators;
    }

    /**
     * Reformat model date from Y-m-d H:i:s to Y-m-d for day comparison
     * return value ready for validateAttribute
     *
     * @param AbstractModel $model
     *
     * @return string
     */
    protected function prepareDataValidation(AbstractModel $model)
    {
        $parsedValue = $this->helper->getFormatDate($this->getValue());
        $this->setValueParsed($parsedValue);

        if (!$model->hasData($this->getEavAttributeCode())) {
            $model->load($model->getId());
        }
        $attributeValue = $model->getData($this->getEavAttributeCode());
        if (!is_string($attributeValue)) {
            return $attributeValue;
        }

        // change date format
        return $this->helper->getFormatDate($attributeValue);
    }

    /**
     * Prepare date diff
     * Reformat model date from Y-m-d H:i:s to Y-m-d for correct day comparison
     * return value ready for validateAttribute
     *
     * @param AbstractModel $model
     * @param bool $isPositiveOffset
     *
     * @return string
     */
    protected function prepareDayValidation(AbstractModel $model, bool $isPositiveOffset = false)
    {
        $parsedValue = $isPositiveOffset
            ? $this->helper->getDateWithPositiveOffset($this->getValue(), 'Y-m-d')
            : $this->helper->getDateDiffFormat($this->getValue(), 'Y-m-d');
        $this->setValueParsed($parsedValue);

        if (!$model->hasData($this->getEavAttributeCode())) {
            $model->load($model->getId());
        }
        $attributeValue = $model->getData($this->getEavAttributeCode());
        if (!is_string($attributeValue)) {
            return $attributeValue;
        }

        // change date format
        return $this->helper->getFormatDate($attributeValue);
    }

    protected function getBirthDateInCurrentYear(AbstractModel $model): ?string
    {
        $result = null;

        if ($attributeValue = $model->getData($this->getEavAttributeCode())) {
            if ($this->isStartUnixEpoch($attributeValue)) {
                $attributeValue = '1970-01-01 00:00:01';
            }

            $birthdayWithCurrentYear = $this->helper->getDateDiffFormat($this->getValue(), 'Y')
                . '-' . $this->helper->getFormatDate($attributeValue, 'm-d');
            $result = $this->helper->getFormatDate($birthdayWithCurrentYear, 'Y-m-d');
        }

        return $result;
    }

    private function isStartUnixEpoch($date): bool
    {
        return strtotime($date) === 0;
    }
}
