<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/

namespace Amasty\Segments\Plugin;

use Amasty\Segments\Controller\Adminhtml\Segment;
use Amasty\Segments\Helper\Base;
use Amasty\Segments\Helper\Customer\Data;
use Amasty\Segments\Model\SegmentRepository;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Convert\Excel;
use Magento\Framework\Convert\ExcelFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Filesystem;
use Magento\Framework\Filesystem\Directory\WriteInterface;
use Magento\Framework\Math\Random;
use Magento\Framework\View\Element\UiComponent\Context;
use Magento\Ui\Component\MassAction\Filter;
use Magento\Ui\Model\Export\MetadataProvider;
use Magento\Ui\Model\Export\SearchResultIterator;
use Magento\Ui\Model\Export\SearchResultIteratorFactory;

abstract class AbstractExport
{
    public const CUSTOMER_COMPONENT_NAMESPACE = 'amastysegments_customer_listing';

    public const GUEST_COMPONENT_NAMESPACE = 'amastysegments_guest_listing';

    /**
     * @var string
     */
    protected $exportType = '';

    /**
     * @var Base
     */
    private $baseHelper;

    /**
     * @var RequestInterface
     */
    protected $request;

    /**
     * @var WriteInterface
     */
    protected $directory;

    /**
     * @var MetadataProvider
     */
    protected $metadataProvider;

    /**
     * @var ExcelFactory
     */
    protected $excelFactory;

    /**
     * @var array
     */
    protected $options;

    /**
     * @var SearchResultIteratorFactory
     */
    protected $iteratorFactory;

    /**
     * @var array
     */
    protected $fields;

    /**
     * @var int|null
     */
    protected $pageSize = null;

    /**
     * @var SegmentRepository
     */
    protected $segmentRepository;

    /**
     * @var Context
     */
    protected $context;

    /**
     * @var Data
     */
    private $customerHelper;

    /**
     * @var Random
     */
    private $random;

    /**
     * @var Filter
     */
    private $filter;

    public function __construct(
        Base $baseHelper,
        RequestInterface $request,
        SegmentRepository $segmentRepository,
        Context $context,
        Filesystem $filesystem,
        Filter $filter,
        MetadataProvider $metadataProvider,
        ExcelFactory $excelFactory,
        SearchResultIteratorFactory $iteratorFactory,
        Data $customerHelper,
        Random $random,
        $pageSize = 200
    ) {
        $this->filter = $filter;
        $this->directory = $filesystem->getDirectoryWrite(DirectoryList::VAR_DIR);
        $this->metadataProvider = $metadataProvider;
        $this->excelFactory = $excelFactory;
        $this->iteratorFactory = $iteratorFactory;
        $this->pageSize = $pageSize;
        $this->baseHelper = $baseHelper;
        $this->segmentRepository = $segmentRepository;
        $this->request = $request;
        $this->context = $context;
        $this->customerHelper = $customerHelper;
        $this->random = $random;
    }

    /**
     * @return bool
     */
    public function checkNamespace()
    {
        $namespace = $this->request->getParam('namespace');

        return (!in_array($namespace, [self::CUSTOMER_COMPONENT_NAMESPACE, self::GUEST_COMPONENT_NAMESPACE]));
    }

    /**
     * @return array
     */
    protected function exportByType()
    {
        $result = '';
        $this->baseHelper->initCurrentSegment(
            $this->segmentRepository->get($this->request->getParam(Segment::SEGMENT_PARAM_URL_KEY, 0))
        );

        switch ($this->exportType) {
            case 'csv':
                $result = $this->getCsvFile();
                break;
            case 'xml':
                $result = $this->getXmlFile();
                break;
        }

        return $result;
    }

    /**
     * Returns CSV file
     *
     * @return array
     * @throws LocalizedException
     */
    public function getCsvFile()
    {
        $component = $this->filter->getComponent();

        $name = $this->random->getRandomString(32);
        $file = 'export/' . $component->getName() . $name . '.csv';

        $this->filter->prepareComponent($component);
        $this->filter->applySelectionOnTargetProvider();
        $items = $this->getExportItems();
        $fields = $this->metadataProvider->getFields($component);

        $this->directory->create('export');
        $stream = $this->directory->openFile($file, 'w+');
        $stream->lock();
        $stream->writeCsv($this->metadataProvider->getHeaders($component));

        foreach ($items as $item) {
            $this->metadataProvider->convertDate($item, $component->getName());
            $stream->writeCsv($this->getRowData($item->getData(), $fields));
        }
        $stream->unlock();
        $stream->close();

        return [
            'type' => 'filename',
            'value' => $file,
            'rm' => true
        ];
    }

    /**
     * @return array
     */
    public function getXmlFile()
    {
        $component = $this->filter->getComponent();

        $name = $this->random->getRandomString(32);
        $file = 'export/' . $component->getName() . $name . '.xml';

        $this->filter->prepareComponent($component);
        $this->filter->applySelectionOnTargetProvider();

        $component->getContext()->getDataProvider()->setLimit(0, 0);
        $dataProvider = $component->getContext()->getDataProvider();
        $dataSource['data']['items'] = $dataProvider->getData()['items'];
        $searchResultItems = $this->prepareDataSource($dataSource, $component);

        /** @var SearchResultIterator $searchResultIterator */
        $searchResultIterator = $this->iteratorFactory->create(['items' => $searchResultItems]);

        /** @var Excel $excel */
        $excel = $this->excelFactory->create([
            'iterator' => $searchResultIterator,
            'rowCallback'=> [$this, 'getRowXmlData'],
        ]);

        $this->directory->create('export');
        $stream = $this->directory->openFile($file, 'w+');
        $stream->lock();

        $excel->setDataHeader($this->metadataProvider->getHeaders($component));
        $excel->write($stream, $component->getName() . '.xml');

        $stream->unlock();
        $stream->close();

        return [
            'type' => 'filename',
            'value' => $file,
            'rm' => true
        ];
    }

    /**
     * @return \Magento\Framework\DataObject[]
     */
    private function getExportItems()
    {
        $selected = $this->request->getParam(Filter::SELECTED_PARAM);
        $excluded = $this->request->getParam(Filter::EXCLUDED_PARAM);

        if ($this->request->getParam('namespace') == self::CUSTOMER_COMPONENT_NAMESPACE) {
            $collection = $this->customerHelper->getFilteredCustomerCollection();
        } else {
            $collection = $this->customerHelper->getFilteredGuestCollection();
        }
        if ($selected !== 'false' || $excluded) {
            $collection = $this->filter->getCollection($collection);
        }
        $items = $collection->getItems();

        return $items;
    }

    /**
     * @param array $data
     * @param \Magento\Framework\View\Element\UiComponentInterface $component
     * @return array
     */
    protected function prepareDataSource(array &$data, \Magento\Framework\View\Element\UiComponentInterface $component)
    {
        $childComponents = $component->getChildComponents();

        if (!empty($childComponents)) {
            foreach ($childComponents as $child) {
                $this->prepareDataSource($data, $child);
            }
        }
        $data = $component->prepareDataSource($data);

        return $data['data']['items'];
    }

    /**
     * @param $item
     * @return array
     */
    public function getRowXmlData($item)
    {
        $fields = $this->metadataProvider->getFields($this->filter->getComponent());
        $options = $this->metadataProvider->getOptions();

        return $this->getRowData($item, $fields);
    }

    /**
     * @param $item
     * @param $fields
     * @param $options
     * @return array
     */
    protected function getRowData($item, $fields)
    {
        $row = [];

        foreach ($fields as $column) {
            if (array_key_exists($column, $item) && $item[$column]) {
                $row[] = $item[$column];
            } else {
                $row[] = '-';
            }
        }

        return $row;
    }
}
