<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/

namespace Amasty\Segments\Model;

use Amasty\Segments\Api\Data\SegmentInterface;
use Amasty\Segments\Helper\Base;
use Amasty\Segments\Model\ResourceModel\Segment as SegmentResource;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;

class Segment extends AbstractModel implements SegmentInterface
{
    public const STATUS_ENABLED = 1;
    public const STATUS_DISABLED = 0;

    /**
     * @var SalesRule
     */
    private $salesRule;

    /**
     * @var SalesRule
     */
    private $salesRuleFactory;

    /**
     * @var Base
     */
    private $baseHelper;

    /**
     * @var SegmentResource
     */
    private $segmentResource;

    public function __construct(
        Context $context,
        Registry $registry,
        SalesRuleFactory $salesRuleFactory,
        Base $baseHelper,
        SegmentResource $segmentResource,
        AbstractResource $resource = null,
        AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
        $this->salesRuleFactory = $salesRuleFactory;
        $this->baseHelper = $baseHelper;
        $this->segmentResource = $segmentResource;
    }

    /**
     * _construct
     */
    protected function _construct()
    {
        $this->_init(SegmentResource::class);
        $this->setIdFieldName(SegmentInterface::SEGMENT_ID);
    }

    /**
     * @return array
     */
    public function getAvailableStatuses(): array
    {
        return [
            self::STATUS_ENABLED => __('Enabled'),
            self::STATUS_DISABLED => __('Disabled')
        ];
    }

    /**
     * @return SalesRule
     */
    public function getSalesRule(): SalesRule
    {
        if (!$this->salesRule) {
            $salesRuleFactory = $this->salesRuleFactory->create();
            $this->salesRule = $salesRuleFactory->load($this->getId());
        }

        return $this->salesRule;
    }

    /**
     * @param string $type
     * @param string $classPath
     * @return $this
     */
    public function addNewEvent(string $type, string $classPath): self
    {
        /** @var SegmentResource $resource */
        $resource = $this->getResource();

        if ($type && $classPath) {
            $conditions = $this->getSalesRule()->getConditions()->getConditions();

            if ($conditions && $this->baseHelper->checkExistConditionInSegment($conditions, $classPath)) {
                $resource->triggerEvent($type, $this->getSegmentId());
            }
        }

        return $this;
    }

    /**
     * @return SegmentResource
     */
    public function getResource(): SegmentResource
    {
        return $this->segmentResource;
    }

    public function getSegmentId()
    {
        return $this->_getData(SegmentInterface::SEGMENT_ID);
    }

    public function setSegmentId($segmentId)
    {
        $this->setData(SegmentInterface::SEGMENT_ID, $segmentId);

        return $this;
    }

    public function getName()
    {
        return $this->_getData(SegmentInterface::NAME);
    }

    public function setName($name)
    {
        $this->setData(SegmentInterface::NAME, $name);

        return $this;
    }

    public function getDescription()
    {
        return $this->_getData(SegmentInterface::DESCRIPTION);
    }

    public function setDescription($description)
    {
        $this->setData(SegmentInterface::DESCRIPTION, $description);

        return $this;
    }

    public function getIsActive()
    {
        return $this->_getData(SegmentInterface::IS_ACTIVE);
    }

    public function setIsActive($isActive)
    {
        $this->setData(SegmentInterface::IS_ACTIVE, $isActive);

        return $this;
    }

    public function getConditionsSerialized()
    {
        return $this->_getData(SegmentInterface::CONDITIONS_SERIALIZED);
    }

    public function setConditionsSerialized($conditionsSerialized)
    {
        $this->setData(SegmentInterface::CONDITIONS_SERIALIZED, $conditionsSerialized);

        return $this;
    }

    public function getCreatedAt()
    {
        return $this->_getData(SegmentInterface::CREATED_AT);
    }

    public function setCreatedAt($createdAt)
    {
        $this->setData(SegmentInterface::CREATED_AT, $createdAt);

        return $this;
    }

    public function getUpdatedAt()
    {
        return $this->_getData(SegmentInterface::UPDATED_AT);
    }

    public function setUpdatedAt($updatedAt)
    {
        $this->setData(SegmentInterface::UPDATED_AT, $updatedAt);

        return $this;
    }

    /**
     * Get conditions field set id.
     *
     * @param string $formName
     * @return string
     */
    public function getConditionsFieldSetId($formName = ''): string
    {
        return $formName . 'rule_conditions_fieldset_' . $this->getId();
    }
}
