<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition\Product;

use Amasty\Segments\Helper\Condition\Data as ConditionHelper;
use Amasty\Segments\Model\Rule\Utils\ConditionAttributesFormatter;
use Amasty\Segments\Traits\MainValidation;
use Magento\CatalogRule\Model\Rule\Condition\Product;
use Magento\Framework\Model\AbstractModel;
use Magento\Rule\Model\Condition\AbstractCondition;
use Magento\Rule\Model\Condition\Context;
use Magento\Wishlist\Model\Item;

class Subselect extends Combine
{
    /**
     * use MainValidation trait
     */
    use MainValidation;

    /**
     * @var string
     */
    protected $type;

    public function __construct(
        Context $context,
        Product $conditionCatalog,
        ConditionAttributesFormatter $attributesFormatter,
        array $data = []
    ) {
        parent::__construct($context, $conditionCatalog, $attributesFormatter, $data);
        $this->setType(ConditionHelper::AMASTY_SEGMENTS_PATH_TO_CONDITIONS . 'Product\Subselect')->setValue(null);
    }

    /**
     * Load array
     *
     * @param array $arr
     * @param string $key
     * @return $this
     */
    public function loadArray($arr, $key = 'conditions'): AbstractCondition
    {
        $this->setAttribute($arr['attribute']);
        $this->setOperator($arr['operator']);
        parent::loadArray($arr, $key);

        return $this;
    }

    /**
     * Return as html
     *
     * @return string
     */
    public function asHtml(): string
    {
        if ($this->type == 'wishlist') {
            $conditionText = __(
                'If %1 %2 %3 for a subselection of product items in %4 matching %5 of these conditions:',
                $this->getAttributeElement()->getHtml(),
                $this->getOperatorElement()->getHtml(),
                $this->getValueElement()->getHtml(),
                $this->type,
                $this->getAggregatorElement()->getHtml()
            );
        } else {
            $conditionText = __(
                'If %1 %2 %3 for a subselection of %4 product matching %5 of these conditions:',
                $this->getAttributeElement()->getHtml(),
                $this->getOperatorElement()->getHtml(),
                $this->getValueElement()->getHtml(),
                $this->type,
                $this->getAggregatorElement()->getHtml()
            );
        }
        $html = $this->getTypeElement()->getHtml() . $conditionText;

        if ($this->getId() != '1') {
            $html .= $this->getRemoveLinkHtml();
        }

        return $html;
    }

    /**
     * Return as xml
     *
     * @param string $containerKey
     * @param string $itemKey
     * @return string
     */
    public function asXml($containerKey = 'conditions', $itemKey = 'condition'): string
    {
        return '<attribute>' . $this->getAttribute() . '</attribute>'
            . '<operator>' . $this->getOperator() . '</operator>'
            . parent::asXml($containerKey, $itemKey);
    }

    /**
     * Load attribute options
     *
     * @return $this
     */
    public function loadAttributeOptions(): AbstractCondition
    {
        $this->setAttributeOption(['qty' => __('total quantity')]);

        return $this;
    }

    /**
     * Load value options
     *
     * @return $this
     */
    public function loadValueOptions(): AbstractCondition
    {
        return $this;
    }

    /**
     * @return \Magento\Reports\Model\ResourceModel\Product\Index\Viewed\Collection
     * |\Magento\Wishlist\Model\ResourceModel\Item\Collection
     */
    public function getValidationCollection($customer)
    {
        return $this;
    }

    /**
     * Load operator options
     *
     * @return $this
     */
    public function loadOperatorOptions(): AbstractCondition
    {
        $this->setOperatorOption(
            [
                '==' => __('is'),
                '!=' => __('is not'),
                '>=' => __('equals or greater than'),
                '<=' => __('equals or less than'),
                '>' => __('greater than'),
                '<' => __('less than'),
                '()' => __('is one of'),
                '!()' => __('is not one of'),
            ]
        );

        return $this;
    }

    /**
     * Get value element type
     *
     * @return string
     */
    public function getValueElementType(): string
    {
        return 'text';
    }

    /**
     * @param AbstractModel $model
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function validate(AbstractModel $model): bool
    {
        $model = $this->objectValidation($model);

        if (!$model || $model->getCustomerIsGuest()) {
            return false;
        }

        $total = 0;

        foreach ($this->getValidationCollection($model) as $item) {
            $qty = 1;

            if ($item instanceof Item) {
                $qty = $item->getQty();
                $item = $item->getProduct();
            }

            if (!$this->getConditions() || parent::validate($item)) {
                $total += $qty;
            }
        }

        return $this->validateAttribute($total);
    }
}
