<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition\Order\Subselect;

use Amasty\Segments\Helper\Condition\Data;
use Amasty\Segments\Helper\Order\Data as OrderHelper;
use Amasty\Segments\Model\Rule\Condition\Order\Common;
use Amasty\Segments\Model\Rule\Condition\Order\Subselect;
use Amasty\Segments\Model\Rule\Utils\ConditionAttributesFormatter;
use Magento\Framework\Model\AbstractModel;
use Magento\Rule\Model\Condition\AbstractCondition;
use Magento\Rule\Model\Condition\Context;
use Magento\SalesRule\Model\Rule\Condition\Product;

class Ordered extends Subselect
{
    /**
     * @var ConditionAttributesFormatter
     */
    private $attributesFormatter;

    public function __construct(
        Context $context,
        Product $conditionProduct,
        Common $conditionCommon,
        OrderHelper $orderHelper,
        ConditionAttributesFormatter $attributesFormatter,
        array $data = []
    ) {
        parent::__construct(
            $context,
            $conditionProduct,
            $conditionCommon,
            $attributesFormatter,
            $orderHelper,
            $data
        );
        $this->setType(Data::AMASTY_SEGMENTS_PATH_TO_CONDITIONS . 'Order\Subselect\Ordered')->setValue(null);
        $this->attributesFormatter = $attributesFormatter;
    }

    /**
     * Load attribute options
     *
     * @return $this
     */
    public function loadAttributeOptions(): AbstractCondition
    {
        $this->setAttributeOption(['qty' => __('total quantity')]);

        return $this;
    }

    /**
     * Return as html
     *
     * @return string
     */
    public function asHtml(): string
    {
        $html = $this->getTypeElement()->getHtml() . __(
            "If %1 %2 %3 for a subselection of products for orders matching %4 of these conditions:",
            $this->getAttributeElement()->getHtml(),
            $this->getOperatorElement()->getHtml(),
            $this->getValueElement()->getHtml(),
            $this->getAggregatorElement()->getHtml()
        );

        if ($this->getId() != '1') {
            $html .= $this->getRemoveLinkHtml();
        }

        return $html;
    }

    /**
     * Get new child select options
     *
     * @return array
     */
    public function getNewChildSelectOptions(): array
    {
        return array_merge_recursive(
            AbstractCondition::getNewChildSelectOptions(),
            [
                [
                    'label' => __('Product Attribute'),
                    'value' => $this->attributesFormatter->format(
                        $this->conditionProduct,
                        ['quote_item_price', 'quote_item_qty', 'quote_item_row_total']
                    )
                ],
            ]
        );
    }

    /**
     * Validate
     *
     * @param AbstractModel $model
     *
     * @return bool
     */
    public function validate(AbstractModel $model): bool
    {
        $total = 0;
        $model = $this->objectValidation($model);
        $orders = $this->orderHelper->getCollectionByCustomerType($model);

        if ($orders && count($orders)) {
            foreach ($orders as $order) {
                $total += $this->getValidItemsTotal($order);
            }

            return $this->validateAttribute($total);
        }

        return false;
    }

    protected function canValidateGuest(): bool
    {
        return true;
    }
}
