<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition\Order;

use Amasty\Segments\Helper\Condition\Data as ConditionHelper;
use Amasty\Segments\Helper\Order\Data;
use Amasty\Segments\Model\Rule\Utils\ConditionAttributesFormatter;
use Amasty\Segments\Traits\MainValidation;
use Magento\Rule\Model\Condition\AbstractCondition;
use Magento\Rule\Model\Condition\Context;
use Magento\SalesRule\Model\Rule\Condition\Product;

class Subselect extends Combine
{
    /**
     * use traits
     */
    use MainValidation;

    /**
     * @var Data
     */
    protected $orderHelper;

    public function __construct(
        Context $context,
        Product $conditionProduct,
        Common $conditionCommon,
        ConditionAttributesFormatter $attributesFormatter,
        Data $orderHelper,
        array $data = []
    ) {
        parent::__construct(
            $context,
            $conditionProduct,
            $conditionCommon,
            $attributesFormatter,
            $data
        );
        $this->setType(ConditionHelper::AMASTY_SEGMENTS_PATH_TO_CONDITIONS . 'Order\Subselect')->setValue(null);
        $this->orderHelper = $orderHelper;
    }

    /**
     * Load array
     *
     * @param array $arr
     * @param string $key
     * @return $this
     */
    public function loadArray($arr, $key = 'conditions'): AbstractCondition
    {
        $this->setAttribute($arr['attribute']);
        $this->setOperator($arr['operator']);
        parent::loadArray($arr, $key);

        return $this;
    }

    /**
     * Return as xml
     *
     * @param string $containerKey
     * @param string $itemKey
     * @return string
     */
    public function asXml($containerKey = 'conditions', $itemKey = 'condition'): string
    {
        return '<attribute>' . $this->getAttribute() . '</attribute>'
            . '<operator>' . $this->getOperator() . '</operator>'
            . parent::asXml($containerKey, $itemKey);
    }

    /**
     * Load attribute options
     *
     * @return $this
     */
    public function loadAttributeOptions(): AbstractCondition
    {
        $this->setAttributeOption(['qty' => __('total quantity'), 'base_row_total' => __('total amount')]);

        return $this;
    }

    /**
     * Load value options
     *
     * @return $this
     */
    public function loadValueOptions(): AbstractCondition
    {
        return $this;
    }

    /**
     * Load operator options
     *
     * @return $this
     */
    public function loadOperatorOptions(): AbstractCondition
    {
        $this->setOperatorOption(
            [
                '==' => __('is'),
                '>=' => __('equals or greater than'),
                '<=' => __('equals or less than'),
                '>' => __('greater than'),
                '<' => __('less than'),
            ]
        );

        return $this;
    }

    /**
     * Get value element type
     *
     * @return string
     */
    public function getValueElementType(): string
    {
        return 'text';
    }

    /**
     * Return as html
     *
     * @return string
     */
    public function asHtml(): string
    {
        $html = $this->getTypeElement()->getHtml() . __(
            "If %1 %2 %3 for a subselection of orders matching %4 of these conditions:",
            $this->getAttributeElement()->getHtml(),
            $this->getOperatorElement()->getHtml(),
            $this->getValueElement()->getHtml(),
            $this->getAggregatorElement()->getHtml()
        );

        if ($this->getId() != '1') {
            $html .= $this->getRemoveLinkHtml();
        }

        return $html;
    }

    protected function getValidItemsTotal($order): int
    {
        $total = 0;

        foreach ($order->getAllVisibleItems() as $item) {
            try {
                if (!$this->getConditions() || parent::validate($item)) {
                    $total++;
                }
            } catch (\Exception $e) {
                null; // Skip if the requested item has been deleted
            }
        }

        return $total;
    }
}
