<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition\Order;

use Amasty\Segments\Helper\Order\Data as OrderHelper;
use Amasty\Segments\Model\Rule\Condition\Condition;
use Amasty\Segments\Traits\MainValidation;
use Magento\Backend\Helper\Data as BackendHelper;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;
use Magento\Rule\Model\Condition\Context;
use Magento\Sales\Model\ResourceModel\Order\Collection;

class CouponUsage extends Condition
{
    use MainValidation;

    /**
     * @var OrderHelper
     */
    private $orderHelper;

    /**
     * @var BackendHelper
     */
    private $backendHelper;

    public function __construct(
        Context $context,
        OrderHelper $orderHelper,
        BackendHelper $adminhtmlData,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->orderHelper = $orderHelper;
        $this->backendHelper = $adminhtmlData;
    }

    /**
     * @return $this
     */
    public function loadAttributeOptions(): self
    {
        $this->setAttributeOption(['coupon_code' => __('Used Coupon')]);

        return $this;
    }

    /**
     * Load operator options
     *
     * @return $this
     */
    public function loadOperatorOptions(): self
    {
        $this->setOperatorOption(
            [
                '==' => __('is'),
                '!=' => __('is not'),
                '()' => __('is one of'),
                '{}' => __('contains'),
                '!{}' => __('does not contain')
            ]
        );

        return $this;
    }

    public function getInputType(): string
    {
        return 'string';
    }

    public function getValueElementType(): string
    {
        return 'text';
    }

    /**
     * Retrieve after element HTML
     *
     * @return string
     */
    public function getValueAfterElementHtml(): string
    {
        $html = '<a href="javascript:void(0)" class="rule-chooser-trigger"><img src="'
            . $this->_assetRepo->getUrl('images/rule_chooser_trigger.gif')
            . '" alt="" class="v-middle rule-chooser-trigger" title="'
            . __('Open Chooser') . '" /></a>';

        return $html;
    }

    /**
     * Retrieve value element chooser URL
     *
     * @return string
     */
    public function getValueElementChooserUrl(): string
    {
        $url = 'amastysegments/widget/chooser/attribute/' . $this->getAttribute();
        if ($this->getJsFormObject()) {
            $url .= '/form/' . $this->getJsFormObject();
        }

        return $this->backendHelper->getUrl($url);
    }

    /**
     * Retrieve Explicit Apply
     *
     * @return bool
     * @SuppressWarnings(PHPMD.BooleanGetMethodName)
     */
    public function getExplicitApply(): bool
    {
        return true;
    }

    /**
     * Present selected values as array
     *
     * @return array
     */
    public function getValueParsed(): array
    {
        $value = $this->getData('value');
        $value = array_map('trim', explode(',', $value));
        $value = array_filter($value);

        if (in_array($this->getOperator(), ['{}', '!{}'])) {
            $value = array_map(
                function ($item) {
                    return '%' . $item . '%';
                },
                $value
            );
        }

        return $value;
    }

    /**
     * Validate model.
     *
     * @param AbstractModel $model
     * @return bool
     */
    public function validate(AbstractModel $model): bool
    {
        $model = $this->objectValidation($model);
        $orderCollection = $this->orderHelper->getCollectionByCustomerType($model);
        if ($orderCollection && $this->getValueParsed()) {
            $this->addCustomerCondition($model, $orderCollection);

            return $this->isNeverUsedCoupons($orderCollection)
                || $this->isUsedCoupons($orderCollection);
        }

        return false;
    }

    /**
     * Checking missing orders of the current customer using coupons for negative conditions:
     *  "is not" and "does not contain".
     *  The segment should include customers who have never used coupons for negative conditions.
     *
     * @param Collection $orderCollection
     * @return bool
     */
    private function isNeverUsedCoupons(Collection $orderCollection): bool
    {
        if (in_array($this->getOperator(), ['!=', '!{}'])) {
            $collection = clone $orderCollection;

            return !$this->validateByCollection(
                $collection->addFieldToFilter('coupon_code', ['notnull' => true])
            );
        }

        return false;
    }

    /**
     * Checking orders of the current customer using coupons by rule condition
     *
     * @param Collection $orderCollection
     * @return bool
     */
    private function isUsedCoupons(Collection $orderCollection): bool
    {
        return $this->validateByCollection(
            $this->applyFilterToCollection($orderCollection, $this->getValueParsed())
        );
    }

    /**
     * Checking if at least one record exists in the collection
     *
     * @param AbstractCollection $collection
     * @return bool
     */
    private function validateByCollection(AbstractCollection $collection): bool
    {
        return (bool)$collection->setPageSize(1)
                ->clear()
                ->count();
    }

    /**
     * Add guest customer condition to order collection
     *
     * @param AbstractModel $model
     * @param Collection $orderCollection
     * @return void
     */
    private function addCustomerCondition(AbstractModel $model, Collection $orderCollection): void
    {
        if ($model->getCustomerIsGuest()) {
            $orderCollection->addFieldToFilter('main_table.quote_id', ['eq' => $model->getQuoteId()]);
        }
    }

    /**
     * Applying a selection condition to a collection based on a rule condition and its values
     *
     * @param AbstractCollection $collection
     * @param array $filterValues
     * @return AbstractCollection
     * @throws InputException
     */
    private function applyFilterToCollection(
        AbstractCollection $collection,
        array $filterValues
    ): AbstractCollection {
        $conditions = [];
        $connection = $collection->getConnection();
        foreach ($filterValues as $filterValue) {
            $conditions[] = $connection->prepareSqlCondition(
                $this->getAttribute(),
                [$this->mapRuleOperatorToCondition($this->getOperator()) => $filterValue]
            );
        }

        $collection->getSelect()->where(implode(
            $this->getOperator() == '()' ? ' OR ' : ' AND ',
            $conditions
        ));

        return $collection;
    }

    /**
     * Maps rule operators to matching operators in a collection filter
     *
     * @param string $ruleOperator
     * @return string
     * @throws InputException
     */
    private function mapRuleOperatorToCondition(string $ruleOperator): string
    {
        $operatorsMap = [
            '==' => 'finset', // Finset used for compatibility with Amasty_Coupons
            '!=' => 'nfinset',
            '()' => 'finset',
            '{}' => 'like',
            '!{}' => 'nlike'
        ];

        if (!array_key_exists($ruleOperator, $operatorsMap)) {
            throw new InputException(
                __(
                    'Undefined rule operator "%1" passed in. Valid operators are: %2',
                    $ruleOperator,
                    implode(',', array_keys($operatorsMap))
                )
            );
        }

        return $operatorsMap[$ruleOperator];
    }

    /**
     * Is this condition can be used for Guest
     *
     * @return bool
     */
    protected function canValidateGuest(): bool
    {
        return true;
    }
}
