<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/

namespace Amasty\Segments\Model\Rule\Condition\Order;

use Amasty\Segments\Helper\Order\Data;
use Amasty\Segments\Model\Rule\Condition\Condition;
use Amasty\Segments\Traits\DayValidation;
use Amasty\Segments\Traits\MainValidation;
use Magento\Framework\Model\AbstractModel;
use Magento\Rule\Model\Condition\Context;
use Magento\Sales\Model\Config\Source\Order\Status;
use Magento\Sales\Model\Order\Item as OrderItem;

class Common extends Condition
{
    /**
     * use traits
     */
    use MainValidation, DayValidation;

    /**
     * @var Status
     */
    private $orderStatus;

    /**
     * @var Data
     */
    private $helper;

    /**
     * @var array
     */
    private $orders = [];

    public function __construct(
        Context $context,
        Status $orderStatus,
        Data $orderHelper,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->orderStatus = $orderStatus;
        $this->helper      = $orderHelper;
    }

    public function loadAttributeOptions(): self
    {
        $attributes = [
            'status'          => __('Order Status'),
            'days_was_placed' => __('Was placed (days) ago'),
            'created_at'      => __('Placed'),
        ];

        $this->setAttributeOption($attributes);

        return $this;
    }

    public function getInputType(): string
    {
        switch ($this->getAttribute()) {
            case 'days_was_placed':
                return 'day';
            case 'status':
                return 'select';
            case 'created_at':
                return 'date';
        }

        return 'string';
    }

    public function getValueElementType(): string
    {
        switch ($this->getAttribute()) {
            case 'status':
                return 'select';
            case 'created_at':
                return 'date';
        }

        return 'text';
    }

    /**
     * @return array
     */
    public function getValueSelectOptions()
    {
        if (!$this->hasData('value_select_options')) {

            switch ($this->getAttribute()) {
                case 'status':
                    $options = $this->orderStatus->toOptionArray();
                    break;

                default:
                    $options = [];
            }
            $this->setData('value_select_options', $options);
        }

        return $this->getData('value_select_options');
    }

    /**
     * @return $this|string
     */
    public function getValueElement()
    {
        $element = parent::getValueElement();

        switch ($this->getInputType()) {
            case 'date':
                $element->setClass('hasDatepicker');
                $element->setExplicitApply(true);
                break;
        }

        return $element;
    }

    protected function getEavAttributeCode(): string
    {
        switch ($this->getAttribute()) {
            case 'days_was_placed':
                return 'created_at';
        }

        return $this->getAttribute();
    }

    /**
     * @param \Magento\Sales\Model\Order|\Magento\Sales\Model\Order\Item $model
     * @return bool
     */
    public function validate(AbstractModel $model): bool
    {
        $model = $this->replaceRuleEntity($model);
        if (!$model) {
            return false;
        }

        switch ($this->getAttribute()) {
            case 'created_at':
                $attributeValue = $this->prepareDataValidation($model);

                return parent::validateAttribute($attributeValue);
            case 'days_was_placed':
                $attributeValue = $this->prepareDayValidation($model);

                return parent::validateAttribute($attributeValue);
        }

        return parent::validate($model);
    }

    private function replaceRuleEntity(AbstractModel $entity): ?AbstractModel
    {
        if ($entity instanceof OrderItem) {
            if (!isset($this->orders[$entity->getOrderId()])) {
                $this->orders[$entity->getOrderId()] = $entity->getOrder();
            }

            return $this->orders[$entity->getOrderId()];
        }

        return $entity;
    }
}
