<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition\Order;

use Amasty\Segments\Model\Rule\Utils\ConditionAttributesFormatter;
use Magento\Catalog\Model\ResourceModel\Collection\AbstractCollection;
use Magento\Catalog\Model\ResourceModel\Product\Collection;
use Amasty\Segments\Helper\Condition\Data as ConditionHelper;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Model\AbstractModel;
use Magento\Rule\Model\Condition\Context;
use Magento\Sales\Model\Order;
use Magento\SalesRule\Model\Rule\Condition\Product;

class Combine extends \Magento\Rule\Model\Condition\Combine
{
    /**
     * @var Product
     */
    protected $conditionProduct;

    /**
     * @var Common
     */
    private $conditionCommon;

    /**
     * @var ConditionAttributesFormatter
     */
    private $attributesFormatter;

    public function __construct(
        Context $context,
        Product $conditionProduct,
        Common $conditionCommon,
        ConditionAttributesFormatter $attributesFormatter,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->setType(ConditionHelper::AMASTY_SEGMENTS_PATH_TO_CONDITIONS . 'Order\Combine');
        $this->conditionProduct = $conditionProduct;
        $this->conditionCommon = $conditionCommon;
        $this->attributesFormatter = $attributesFormatter;
    }

    /**
     * Get new child select options
     *
     * @return array
     */
    public function getNewChildSelectOptions(): array
    {
        return array_merge_recursive(
            parent::getNewChildSelectOptions(),
            [
                [
                    'label' => __('Product Attribute'),
                    'value' => $this->attributesFormatter->format(
                        $this->conditionProduct,
                        ['quote_item_price', 'quote_item_qty', 'quote_item_row_total']
                    )
                ],
                ['label' => __('Common'), 'value' => $this->attributesFormatter->format($this->conditionCommon)]
            ]
        );
    }

    /**
     * Collect validated attributes
     *
     * @param Collection $productCollection
     * @return $this
     */
    public function collectValidatedAttributes(AbstractCollection $productCollection): Combine
    {
        foreach ($this->getConditions() as $condition) {
            $condition->collectValidatedAttributes($productCollection);
        }

        return $this;
    }

    /**
     * Is entity valid
     *
     * @param int|AbstractModel $entity
     * @return bool
     * @throws NoSuchEntityException
     */
    protected function _isValid($entity): bool
    {
        if (!$this->getConditions()) {
            return true;
        }

        $all = $this->getAggregator() === 'all';
        $true = (bool)$this->getValue();
        $validated = false;

        foreach ($this->getConditions() as $cond) {

            if ($cond instanceof Product) {
                if ($entity instanceof Order) {
                    $allItems = $entity->getAllVisibleItems();
                    $validated = false;

                    foreach ($allItems as $item) {
                        //Magento\ConfigurableProduct\Plugin\SalesRule\Model\Rule\Condition\Product workaround
                        $item->setChildren($item->getChildrenItems());

                        if ($item->getProduct() && $cond->validate($item)) {
                            $validated = true;

                            break;
                        }
                    }
                } elseif ($entity instanceof \Magento\Sales\Model\Order\Item) {
                    //Magento\ConfigurableProduct\Plugin\SalesRule\Model\Rule\Condition\Product workaround
                    $entity->setChildren($entity->getChildrenItems());
                    $validated = $cond->validate($entity);
                } elseif ($entity->getProduct()) {
                    $validated = $cond->validate($entity);
                }
            } else {
                if ($entity instanceof AbstractModel) {
                    $validated = $cond->validate($entity);
                } else {
                    $validated = $cond->validateByEntityId($entity);
                }
            }

            if ($all && $validated !== $true) {
                return false;
            } elseif (!$all && $validated === $true) {
                return true;
            }
        }

        return (bool)$all;
    }
}
