<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition;

use Amasty\Segments\Helper\Base;
use Amasty\Segments\Helper\Condition\Data as ConditionHelper;
use Amasty\Segments\Helper\Order\Data;
use Amasty\Segments\Model\Rule\Condition\Order\CouponUsage;
use Amasty\Segments\Model\Rule\Utils\ConditionAttributesFormatter;
use Amasty\Segments\Traits\DayValidation;
use Amasty\Segments\Traits\MainValidation;
use Magento\Config\Model\Config\Source\Yesno;
use Magento\Eav\Model\Entity\Attribute\AbstractAttribute;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Module\Manager;
use Magento\Framework\ObjectManagerInterface;
use Magento\Payment\Model\Config\Source\Allmethods as PaymentAllmethods;
use Magento\Rule\Model\Condition\AbstractCondition;
use Magento\Rule\Model\Condition\Context;
use Magento\Sales\Model\ResourceModel\Order\Collection as OrderCollection;
use Magento\Shipping\Model\Config\Source\Allmethods;

class Order extends Condition
{
    /**
     * use traits
     */
    use MainValidation, DayValidation;

    /**
     * @var Allmethods
     */
    protected $shippingAllmethods;

    /**
     * @var PaymentAllmethods
     */
    protected $paymentAllmethods;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var Yesno
     */
    private $yesnoOptions;

    /**
     * @var Manager
     */
    private $moduleManager;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var ConditionAttributesFormatter
     */
    private $attributesFormatter;

    /**
     * @var CouponUsage
     */
    private $conditionCoupon;

    public function __construct(
        Context $context,
        Allmethods $shippingAllmethods,
        PaymentAllmethods $paymentAllmethods,
        Data $orderHelper,
        Yesno $yesnoOptions,
        Manager $moduleManager,
        ObjectManagerInterface $objectManager,
        ConditionAttributesFormatter $attributesFormatter,
        CouponUsage $conditionCoupon,
        array $data = []
    ) {
        $this->shippingAllmethods = $shippingAllmethods;
        $this->paymentAllmethods = $paymentAllmethods;
        $this->helper = $orderHelper;
        $this->yesnoOptions = $yesnoOptions;
        $this->moduleManager = $moduleManager;
        $this->objectManager = $objectManager;
        $this->conditionCoupon = $conditionCoupon;
        $this->attributesFormatter = $attributesFormatter;
        parent::__construct($context, $data);
    }

    public function getNewChildSelectOptions(): array
    {
        return array_merge(
            $this->attributesFormatter->format($this),
            $this->attributesFormatter->format($this->conditionCoupon),
            [
                [
                    'value' => ConditionHelper::AMASTY_SEGMENTS_PATH_TO_CONDITIONS . 'Order\Subselect\Quantity',
                    'label' => __('Orders Quantity by Condition'),
                ],
                [
                    'value' => ConditionHelper::AMASTY_SEGMENTS_PATH_TO_CONDITIONS . 'Order\Subselect\Amount',
                    'label' => __('Total Amount by Condition'),
                ],
                [
                    'value' => ConditionHelper::AMASTY_SEGMENTS_PATH_TO_CONDITIONS . 'Order\Subselect\AverageTotal',
                    'label' => __('Average Order Total by Condition'),
                ],
                [
                    'value' => ConditionHelper::AMASTY_SEGMENTS_PATH_TO_CONDITIONS . 'Order\Subselect\Ordered',
                    'label' => __('Ordered Products by Condition'),
                ]
            ]
        );
    }

    /**
     * Retrieve attribute object
     *
     * @return null|AbstractAttribute
     */
    public function getAttributeObject(): ?AbstractAttribute
    {
        if ($this->moduleManager->isEnabled('Amasty_Orderattr')) {
            return $this->helper->getOrderAttribute($this->getAttribute());
        }

        return null;
    }

    /**
     * @return $this
     */
    public function loadAttributeOptions(): AbstractCondition
    {
        $attributes = [
            'days_first_completed' => __('Days From First Completed Order'),
            'days_last_completed' => __('Days From Last Completed Order'),
            'payment_method' => __('Used Payment Methods'),
            'shipping_method' => __('Used Shipping Methods'),
        ];
        /**
         * validation by setting set attributes
         */
        $selectedAttrInConfig = $this->helper->getConfigValueByPath(Base::CONFIG_PATH_GENERAL_ORDER_ATTRIBUTES);

        if ($selectedAttrInConfig) {
            $orderAttributes = $this->helper->getOrderAttributesForSource();
            /** @var AbstractAttribute $attribute */
            foreach ($orderAttributes as $attribute) {

                if (!($attribute->getFrontendLabel()) || !($attribute->getAttributeCode())) {
                    continue;
                }

                if (in_array($attribute->getAttributeCode(), explode(',', $selectedAttrInConfig))) {
                    $attributes[$attribute->getAttributeCode()] = $attribute->getFrontendLabel();
                }
            }
        }
        asort($attributes);
        $this->setAttributeOption($attributes);

        return $this;
    }

    /**
     * @return $this
     */
    public function getAttributeElement()
    {
        $element = parent::getAttributeElement();
        $element->setShowAsText(true);

        return $element;
    }

    /**
     * @return string
     */
    public function getInputType(): string
    {
        switch ($this->getAttribute()) {
            case 'days_first_completed':
            case 'days_last_completed':
                return 'day';

            case 'shipping_method':
            case 'payment_method':
                return 'select';
        }
        $orderAttribute = $this->getAttributeObject();

        if (!$orderAttribute) {
            return parent::getInputType();
        }

        return $this->getInputTypeFromAttribute($orderAttribute);
    }

    /**
     * @param AbstractAttribute $orderAttribute
     * @return mixed|string
     */
    protected function getInputTypeFromAttribute($orderAttribute)
    {
        if (!is_object($orderAttribute)) {
            $orderAttribute = $this->getAttributeObject();
        }

        $possibleTypes = ['string', 'numeric', 'date', 'select', 'multiselect', 'grid', 'boolean'];
        if (in_array($orderAttribute->getFrontendInput(), $possibleTypes)) {
            return $orderAttribute->getFrontendInput();
        }

        switch ($orderAttribute->getFrontendInput()) {
            case 'gallery':
            case 'media_image':
            case 'radios':
            case 'selectimg':
                return 'select';
            case 'multiselectimg':
            case 'checkboxes':
                return 'multiselect';
        }

        return 'string';
    }

    /**
     * @return string
     */
    public function getValueElementType(): string
    {
        switch ($this->getAttribute()) {
            case 'shipping_method':
            case 'payment_method':
                return 'select';
        }
        $orderAttribute = $this->getAttributeObject();

        if (!is_object($orderAttribute)) {
            return parent::getValueElementType();
        }
        $availableTypes = [
            'checkbox',
            'date',
            'editablemultiselect',
            'editor',
            'fieldset',
            'file',
            'gallery',
            'image',
            'imagefile',
            'multiline',
            'multiselect',
            'radio',
            'select',
            'text',
            'textarea',
            'time'
        ];

        if (in_array($orderAttribute->getFrontendInput(), $availableTypes)) {
            return $orderAttribute->getFrontendInput();
        }

        switch ($orderAttribute->getFrontendInput()) {
            case 'radios':
            case 'selectimg':
            case 'boolean':
                return 'select';
            case 'multiselectimg':
            case 'checkboxes':
                return 'multiselect';
        }

        return parent::getValueElementType();
    }

    /**
     * @return array
     * @throws LocalizedException
     */
    public function getValueSelectOptions(): array
    {
        if (!$this->hasData('value_select_options')) {
            switch ($this->getAttribute()) {
                case 'shipping_method':
                    $options = $this->shippingAllmethods->toOptionArray();
                    break;
                case 'payment_method':
                    $options = $this->paymentAllmethods->toOptionArray();
                    break;
                default:
                    $options = [];
                    $attributeObject = $this->getAttributeObject();

                    if (is_object($attributeObject) && $attributeObject->usesSource()) {
                        $addEmptyOption = true;
                        if ($attributeObject->getFrontendInput() == 'multiselect') {
                            $addEmptyOption = false;
                        }
                        $options = $attributeObject->getSource()->getAllOptions($addEmptyOption);
                    }

                    if ($this->getInputType() == 'boolean' && count($options) == 0) {
                        $options = $this->yesnoOptions->toOptionArray();
                    }
                    break;
            }
            $this->setData('value_select_options', $options);
        }

        return (array)$this->getData('value_select_options');
    }

    /**
     * Return real Order attribute for validate
     *
     * @return string
     */
    protected function getEavAttributeCode(): string
    {
        switch ($this->getAttribute()) {
            case 'days_first_completed':
            case 'days_last_completed':
                return 'updated_at';
        }

        return $this->getAttribute();
    }

    /**
     * @param AbstractModel $model
     * @return bool
     * @throws LocalizedException
     */
    public function validate(AbstractModel $model)
    {
        $model = $this->objectValidation($model);
        $orderCollection = $this->helper->getCollectionByCustomerType($model);

        if (!$orderCollection) {
            return false;
        }
        $attribute = $this->getAttribute();

        switch ($attribute) {
            case 'payment_method':
            case 'shipping_method':
                return $this->validateOrderByMethods($orderCollection);
            case 'days_first_completed':
            case 'days_last_completed':
                return $this->validateOrderByAttribute($orderCollection);
        }

        if ($this->moduleManager->isEnabled('Amasty_Orderattr')) {
            $orderCollection->addFieldToFilter('state', ['eq' => \Magento\Sales\Model\Order::STATE_COMPLETE]);

            foreach ($orderCollection->getItems() as $order) {
                try {
                    $orderAttributeData = $this->objectManager
                        ->create(\Amasty\Orderattr\Model\Order\Attribute\Value::class)
                        ->loadByOrderId($order->getId());
                } catch (\ReflectionException $e) {
                    $orderAttributeData = $this->objectManager
                        ->get(\Amasty\Orderattr\Model\Entity\EntityResolver::class)
                        ->getEntityByOrder($order);
                }
                if (parent::validateAttribute(
                    $orderAttributeData->getData($attribute)
                )) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * @param OrderCollection $orderCollection
     * @return bool
     * @throws LocalizedException
     */
    private function validateOrderByMethods(OrderCollection $orderCollection): bool
    {
        /** @var \Magento\Sales\Model\Order $order */
        foreach ($orderCollection->getItems() as $order) {
            if ('payment_method' == $this->getAttribute()) {
                $value = $order->getPayment()->getMethodInstance()->getCode();
                $order->setData($this->getAttribute(), $value);
            }

            if (parent::validate($order)) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param OrderCollection $orderCollection
     *
     * @return bool
     */
    private function validateOrderByAttribute(OrderCollection $orderCollection): bool
    {
        $orderCollection->addFieldToFilter('state', ['eq' => \Magento\Sales\Model\Order::STATE_COMPLETE]);

        /** @var \Magento\Sales\Model\Order $order */
        switch ($this->getAttribute()) {
            case 'days_first_completed':
                $orderCollection->setOrder('updated_at', OrderCollection::SORT_ORDER_DESC);
                break;
            case 'days_last_completed':
                $orderCollection->setOrder('updated_at', OrderCollection::SORT_ORDER_ASC);
                break;
        }
        $order = $orderCollection->setPage(1, 1)->getFirstItem();
        $attributeValue = $this->prepareDayValidation($order);

        return parent::validateAttribute($attributeValue);
    }

    protected function canValidateGuest(): bool
    {
        return true;
    }
}
