<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition;

use Amasty\Segments\Helper\Base;
use Amasty\Segments\Helper\Customer\Data;
use Amasty\Segments\Model\Rule\Condition\Customer\NewsletterStatusOptionsProvider;
use Amasty\Segments\Model\Rule\Utils\ConditionAttributesFormatter;
use Amasty\Segments\Traits\DayValidation;
use Amasty\Segments\Traits\MainValidation;
use Magento\Config\Model\Config\Source\Yesno;
use Magento\Customer\Model\Customer as CustomerModel;
use Magento\Customer\Model\CustomerFactory;
use Magento\Customer\Model\Logger;
use Magento\Customer\Model\ResourceModel\Customer as CustomerResource;
use Magento\Eav\Model\Entity\Attribute\AbstractAttribute;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\AbstractModel;
use Magento\Newsletter\Model\Subscriber;
use Magento\Rule\Model\Condition\AbstractCondition;
use Magento\Rule\Model\Condition\Context;

class Customer extends Condition
{
    /**
     * use Traits
     */
    use MainValidation, DayValidation;

    /**
     * @var CustomerResource
     */
    protected $customerResource;

    /**
     * @var CustomerFactory
     */
    protected $customerFactory;

    /**
     * @var array
     */
    protected $allowedAttributes = [
        'website_id', 'store_id', 'created_at', 'created_in', 'group_id', 'dob', 'disable_auto_group_change', 'email',
        'firstname', 'gender', 'confirmation', 'lastname', 'middlename', 'prefix', 'suffix', 'taxvat'
    ];

    /**
     * @var array
     */
    protected $specialAttributes;

    /**
     * @var NewsletterStatusOptionsProvider
     */
    protected $newsletterOptionsProvider;

    /**
     * @var Subscriber
     */
    protected $subscriber;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var Logger
     */
    protected $customerLogger;

    /**
     * @var Yesno
     */
    private $yesnoOptions;

    /**
     * @var ConditionAttributesFormatter
     */
    private $attributesFormatter;

    public function __construct(
        Context $context,
        CustomerResource $customerResource,
        CustomerFactory $customerFactory,
        NewsletterStatusOptionsProvider $newsletterOptionsProvider,
        Subscriber $subscriber,
        Data $helper,
        Logger $customerLogger,
        Yesno $yesnoOptions,
        ConditionAttributesFormatter $attributesFormatter,
        array $data = []
    ) {
        $this->customerResource = $customerResource;
        $this->customerFactory = $customerFactory;
        $this->newsletterOptionsProvider = $newsletterOptionsProvider;
        $this->subscriber = $subscriber;
        $this->helper = $helper;
        $this->customerLogger = $customerLogger;
        $this->yesnoOptions = $yesnoOptions;
        $this->specialAttributes = [
            'is_newsletter_subscribe' => __('Is Newsletter Subscriber'),
            'day_from_last_visit' => __('Days From the Last Visit'),
            'day_from_registration' => __('Days From the Registration'),
            'day_before_birthday' => __('Days Before Birthday'),
            'day_after_birthday' => __('Days After Birthday'),
        ];
        $this->attributesFormatter = $attributesFormatter;
        parent::__construct($context, $data);
    }

    public function getNewChildSelectOptions(): array
    {
        return $this->attributesFormatter->format($this);
    }

    /**
     * Retrieve attribute object
     *
     * @return bool|null|AbstractAttribute
     * @throws LocalizedException
     */
    public function getAttributeObject()
    {
        return $this->customerResource->getAttribute($this->getAttribute());
    }

    /**
     * @return $this
     */
    public function loadAttributeOptions(): AbstractCondition
    {
        $customerAttributes = $this->customerResource
            ->loadAllAttributes()
            ->getAttributesByCode();
        /**
         * validation by setting set attributes
         */
        $selectedAttrInConfig = $this->helper->getConfigValueByPath(
            Base::CONFIG_PATH_GENERAL_CUSTOMER_ATTRIBUTES
        );
        $attributes = [];

        if ($selectedAttrInConfig) {
            /** @var AbstractAttribute $attribute */
            foreach ($customerAttributes as $attribute) {
                if (!($attribute->getFrontendLabel()) || !($attribute->getAttributeCode())) {
                    continue;
                }

                if (in_array($attribute->getAttributeCode(), explode(',', $selectedAttrInConfig))) {
                    $attributes[$attribute->getAttributeCode()] = $attribute->getFrontendLabel();
                }
            }
        }
        $this->_addSpecialAttributes($attributes);
        asort($attributes);
        $this->setAttributeOption($attributes);

        return $this;
    }

    /**
     * @param array $attributes
     */
    protected function _addSpecialAttributes(array &$attributes): void
    {
        $attributes = array_merge_recursive($attributes, $this->specialAttributes);
    }

    /**
     * @return $this
     */
    public function getAttributeElement()
    {
        $element = parent::getAttributeElement();
        $element->setShowAsText(true);

        return $element;
    }

    /**
     * This value will define which operators will be available for this condition.
     * Possible values are: string, numeric, date, select, multiselect, grid, boolean
     *
     * @return string
     * @throws LocalizedException
     */
    public function getInputType(): string
    {
        switch ($this->getAttribute()) {
            case 'day_from_last_visit':
            case 'day_from_registration':
            case 'day_before_birthday':
            case 'day_after_birthday':
                return 'day';
            case 'is_newsletter_subscribe':
                return 'select';
        }
        $customerAttribute = $this->getAttributeObject();

        if (!$customerAttribute) {
            return parent::getInputType();
        }

        return $this->getInputTypeFromAttribute($customerAttribute);
    }

    /**
     * @param AbstractAttribute $customerAttribute
     *
     * @return string
     * @throws LocalizedException
     */
    protected function getInputTypeFromAttribute(AbstractAttribute $customerAttribute): string
    {
        if (!is_object($customerAttribute)) {
            $customerAttribute = $this->getAttributeObject();
        }
        $possibleTypes = ['string', 'numeric', 'date', 'select', 'multiselect', 'grid', 'boolean'];

        if (in_array($customerAttribute->getFrontendInput(), $possibleTypes)) {
            return $customerAttribute->getFrontendInput();
        }

        switch ($customerAttribute->getFrontendInput()) {
            case 'gallery':
            case 'media_image':
            case 'selectimg':
            case 'radios':
                return 'select';
            case 'multiselectimg':
            case 'checkboxes':
                return 'multiselect';
        }

        return 'string';
    }

    /**
     * @return $this
     * @throws LocalizedException
     */
    public function getValueElement()
    {
        $element = parent::getValueElement();

        if ($this->getInputType() == 'date') {
            $element->setClass('hasDatepicker');
        }

        return $element;
    }

    /**
     * @return bool
     * @throws LocalizedException
     */
    public function getExplicitApply(): bool
    {
        return ($this->getInputType() == 'date');
    }

    /**
     * Value element type will define renderer for condition value element
     *
     * @return string
     * @throws LocalizedException
     * @see \Magento\Framework\Data\Form\Element
     */
    public function getValueElementType(): string
    {
        $customerAttribute = $this->getAttributeObject();

        if ($this->getAttribute() == 'is_newsletter_subscribe') {
            return 'select';
        }

        if (!is_object($customerAttribute)) {
            return parent::getValueElementType();
        }
        $availableTypes = [
            'checkbox',
            'date',
            'editablemultiselect',
            'editor',
            'fieldset',
            'file',
            'gallery',
            'image',
            'imagefile',
            'multiline',
            'multiselect',
            'radio',
            'select',
            'text',
            'textarea',
            'time'
        ];
        if (in_array($customerAttribute->getFrontendInput(), $availableTypes)) {
            return $customerAttribute->getFrontendInput();
        }

        switch ($customerAttribute->getFrontendInput()) {
            case 'selectimg':
            case 'radios':
            case 'boolean':
                return 'select';
            case 'multiselectimg':
            case 'checkboxes':
                return 'multiselect';
        }

        return parent::getValueElementType();
    }

    /**
     * @return array
     * @throws LocalizedException
     */
    public function getValueSelectOptions(): array
    {
        $selectOptions = [];
        $attributeObject = $this->getAttributeObject();

        if (is_object($attributeObject) && $attributeObject->usesSource()) {
            $addEmptyOption = true;

            if ($attributeObject->getFrontendInput() == 'multiselect') {
                $addEmptyOption = false;
            }
            $selectOptions = $attributeObject->getSource()->getAllOptions($addEmptyOption);
        }

        if ($this->getInputType() == 'boolean' && count($selectOptions) == 0) {
            $selectOptions = $this->yesnoOptions->toOptionArray();
        }

        if ($this->getAttribute() == 'is_newsletter_subscribe') {
            $selectOptions = $this->newsletterOptionsProvider->toOptionArray();
        }
        $key = 'value_select_options';

        if (!$this->hasData($key)) {
            $this->setData($key, $selectOptions);
        }

        return (array)$this->getData($key);
    }

    /**
     * Return real Customer attribute code for validate
     *
     * @return string
     */
    protected function getEavAttributeCode(): string
    {
        switch ($this->getAttribute()) {
            case 'day_from_last_visit':
                return 'last_visit_at';
            case 'day_from_registration':
                return 'created_at';
            case 'day_before_birthday':
            case 'day_after_birthday':
                return 'dob';
        }

        return $this->getAttribute();
    }

    /**
     * Validate Address Rule Condition
     *
     * @param AbstractModel $model
     *
     * @return bool
     */
    public function validate(AbstractModel $model): bool
    {
        $customer = $this->objectValidation($model);

        if (!$customer) {
            return false;
        }

        if (array_key_exists($this->getAttribute(), $this->specialAttributes)) {
            return $this->validationByAttribute($customer);
        }

        if (!$customer instanceof CustomerModel) {
            $customer = $model->getQuote()->getCustomer();
            $attr = $this->getAttribute();

            $allAttr = $customer->toArray();

            if ($attr != 'entity_id' && !array_key_exists($attr, $allAttr)) {
                $address = $model->getQuote()->getBillingAddress();
                $allAttr[$attr] = $address->getData($attr);
            }
            $customer = $this->customerFactory->create()->setData($allAttr);
        }

        return parent::validate($customer);
    }

    /**
     * @param \Magento\Customer\Model\Customer|\Amasty\Segments\Model\GuestCustomerData $customer
     * @return bool
     */
    public function validationByAttribute($customer): bool
    {
        if ($this->getAttribute() && $customer->getId()) {
            switch ($this->getAttribute()) {
                case 'day_from_last_visit':
                    $lastLoggedAt = $this->customerLogger->get($customer->getId());
                    if (!$lastLoggedAt->getLastVisitAt()) {
                        return false;
                    }
                    $customer->setData($this->getEavAttributeCode(), $lastLoggedAt->getLastVisitAt());
                    $attributeValue = $this->prepareDayValidation($customer);

                    return $this->validateAttribute($attributeValue);
                case 'day_before_birthday':
                    $birthdayInCurrentYear = $this->getBirthDateInCurrentYear($customer);
                    $customer->setData($this->getEavAttributeCode(), $birthdayInCurrentYear);
                    $attributeValue = $this->prepareDayValidation($customer, true);

                    return ($attributeValue >= $this->helper->getCurrentDate())
                        && $this->validateAttribute($attributeValue);
                case 'day_after_birthday':
                    $birthdayInCurrentYear = $this->getBirthDateInCurrentYear($customer);
                    $customer->setData($this->getEavAttributeCode(), $birthdayInCurrentYear);
                    $attributeValue = $this->prepareDayValidation($customer);

                    return ($attributeValue <= $this->helper->getCurrentDate())
                        && $this->validateAttribute($attributeValue);
                case 'day_from_registration':
                    $attributeValue = $this->prepareDayValidation($customer);

                    return $this->validateAttribute($attributeValue);
                case 'is_newsletter_subscribe':
                    $subscriber = $this->subscriber->loadByCustomer(
                        (int)$customer->getId(),
                        (int)$customer->getWebsiteId()
                    );

                    if ($subscriber->getData()) {
                        return $this->validateAttribute($subscriber->getStatus());
                    }
                    return false;
                default:
                    break;
            }
        }

        return false;
    }
}
