<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition;

use Magento\Framework\Exception\NotFoundException;
use Magento\Framework\ObjectManagerInterface;
use Magento\Rule\Model\Condition\AbstractCondition;

class CombineConditionsProvider
{
    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var array
     */
    private $conditions = [];

    /**
     * @var array
     */
    private $loadedConditionsStorage = [];

    public function __construct(
        ObjectManagerInterface $objectManager,
        array $conditions = []
    ) {
        $this->initializeConditions($conditions);
        $this->objectManager = $objectManager;
    }

    public function getAll(): array
    {
        $result = [];

        foreach (array_keys($this->conditions) as $conditionKey) {
            $result[] = $this->getByKey($conditionKey);
        }

        return $result;
    }

    public function getByKey(string $key): array
    {
        if (!isset($this->conditions[$key])) {
            throw new NotFoundException(
                __('The "%1" condition isn\'t defined.', $key)
            );
        }

        if (!isset($this->loadedConditionsStorage[$key])) {
            /** @var AbstractCondition $conditionObject */
            $conditionObject = $this->objectManager->get($this->conditions[$key]['class']);
            $conditionLabel = $this->conditions[$key]['label'];

            $this->loadedConditionsStorage[$key] = [
                'label' => $conditionLabel,
                'value' => $conditionObject->getNewChildSelectOptions()
            ];
        }

        return $this->loadedConditionsStorage[$key];
    }

    private function initializeConditions(array $conditions): void
    {
        foreach ($conditions as $key => $condition) {
            if (!isset($condition['label']) || !isset($condition['class'])) {
                unset($conditions[$key]);
                continue;
            }
            if (!is_subclass_of($condition['class'], AbstractCondition::class)) {
                throw new \LogicException(
                    sprintf('Condition "%s" must extend %s', $key, AbstractCondition::class)
                );
            }
        }
        uasort($conditions, function ($first, $second) {
            return ($first['sortOrder'] ?? 0) <=> ($second['sortOrder'] ?? 0);
        });

        $this->conditions = $conditions;
    }
}
