<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition;

use Amasty\Segments\Helper\Base;
use Amasty\Segments\Model\GuestCustomerData;
use Amasty\Segments\Model\Rule\Utils\ConditionAttributesFormatter;
use Amasty\Segments\Traits\DayValidation;
use Amasty\Segments\Traits\MainValidation;
use Magento\Framework\Model\AbstractModel;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\ResourceModel\Quote\Collection;
use Magento\Quote\Model\ResourceModel\Quote\CollectionFactory;
use Magento\Rule\Model\Condition\AbstractCondition;
use Magento\Rule\Model\Condition\Context;
use Psr\Log\LoggerInterface;

class Cart extends Condition
{
    /**
     * use MainValidation trait
     */
    use MainValidation, DayValidation;

    /**
     * @var Base
     */
    protected $helper;

    /**
     * @var CartRepositoryInterface
     */
    protected $cartManagement;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var CollectionFactory
     */
    private $quoteCollectionFactory;

    /**
     * @var ConditionAttributesFormatter
     */
    private $attributesFormatter;

    public function __construct(
        Context $context,
        Base $baseHelper,
        CartRepositoryInterface $cartManagement,
        CollectionFactory $quoteCollection,
        ConditionAttributesFormatter $attributesFormatter,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->helper = $baseHelper;
        $this->cartManagement = $cartManagement;
        $this->logger = $context->getLogger();
        $this->quoteCollectionFactory = $quoteCollection;
        $this->attributesFormatter = $attributesFormatter;
    }

    public function getNewChildSelectOptions()
    {
        return $this->attributesFormatter->format($this);
    }

    /**
     * @return $this
     */
    public function loadAttributeOptions(): AbstractCondition
    {
        $attributes = [
            'created_at' => __('Days From Cart Created'),
            'updated_at' => __('Days From Cart Modified'),
            'base_grand_total' => __('Grand Total'),
            'items_qty' => __('Products Count'),
        ];
        $this->setAttributeOption($attributes);

        return $this;
    }

    /**
     * @return string
     */
    public function getInputType(): string
    {
        switch ($this->getAttribute()) {
            case 'created_at':
            case 'updated_at':
                return 'day';
        }

        return 'numeric';
    }

    /**
     * @param \Magento\Customer\Model\Customer|GuestCustomerData $model
     * @return bool
     */
    public function validate(AbstractModel $model): bool
    {
        $model = $this->objectValidation($model);

        if (!$model
            || ($model->getCustomerIsGuest() && !$model->getIsActive())
        ) {
            return false;
        }

        try {
            if ($model->getCustomerIsGuest()) {
                $quote = $model;
            } else {
                $quote = $this->quoteCollectionFactory->create()
                    ->addFieldToFilter('customer_id', $model->getId())
                    ->addFieldToFilter('is_active', 1)
                    ->setOrder('updated_at', Collection::SORT_ORDER_DESC)
                    ->setPageSize(1)
                    ->getFirstItem();
            }

            if ($model && $quote->getId()) {
                if ($this->getInputType() == 'day') {
                    $attributeValue = $this->prepareDayValidation($quote);

                    return parent::validateAttribute($attributeValue);
                }

                return parent::validate($quote);
            }
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }

        return false;
    }

    protected function canValidateGuest(): bool
    {
        return true;
    }
}
