<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/
declare(strict_types=1);

namespace Amasty\Segments\Model\Rule\Condition;

use Amasty\Segments\Model\Rule\Utils\ConditionAttributesFormatter;
use Amasty\Segments\Traits\MainValidation;
use Magento\Directory\Model\Config\Source\Allregion;
use Magento\Directory\Model\Config\Source\Country;
use Magento\Framework\Model\AbstractModel;
use Magento\Rule\Model\Condition\AbstractCondition;
use Magento\Rule\Model\Condition\Context;

class Address extends Condition
{
    /**
     * use MainValidation trait
     */
    use MainValidation;

    /**
     * @var string
     */
    protected $type = '';

    /**
     * @var Country
     */
    private $directoryCountry;

    /**
     * @var Allregion
     */
    private $directoryAllregion;

    /**
     * @var ConditionAttributesFormatter
     */
    protected $attributesFormatter;

    public function __construct(
        Context $context,
        Country $directoryCountry,
        Allregion $directoryAllregion,
        ConditionAttributesFormatter $attributesFormatter,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->directoryCountry = $directoryCountry;
        $this->directoryAllregion = $directoryAllregion;
        $this->attributesFormatter = $attributesFormatter;
    }

    public function getNewChildSelectOptions(): array
    {
        return $this->attributesFormatter->format($this);
    }

    /**
     * Load attribute options
     *
     * @return $this
     */
    public function loadAttributeOptions(): AbstractCondition
    {
        $attributes = [
            'postcode' => __('Shipping Postcode'),
            'region_id' => __('Shipping State/Province'),
            'city' => __('Shipping City'),
            'country_id' => __('Shipping Country'),
        ];
        $this->setAttributeOption($attributes);

        return $this;
    }

    /**
     * Get input type
     *
     * @return string
     */
    public function getInputType(): string
    {
        switch ($this->getAttribute()) {
            case 'postcode':
                return 'numeric';
            case 'country_id':
            case 'region_id':
                return 'select';
        }

        return 'string';
    }

    /**
     * Get value element type
     *
     * @return string
     */
    public function getValueElementType(): string
    {
        switch ($this->getAttribute()) {
            case 'country_id':
            case 'region_id':
                return 'select';
        }

        return 'text';
    }

    /**
     * Get value select options
     *
     * @return array
     */
    public function getValueSelectOptions(): array
    {
        if (!$this->hasData('value_select_options')) {
            switch ($this->getAttribute()) {
                case 'country_id':
                    $options = $this->directoryCountry->toOptionArray();
                    break;
                case 'region_id':
                    $options = $this->directoryAllregion->toOptionArray();
                    break;
                default:
                    $options = [];
            }
            $this->setData('value_select_options', $options);
        }

        return (array)$this->getData('value_select_options');
    }

    /**
     * Default operator input by type map getter
     *
     * @return array
     */
    public function getDefaultOperatorInputByType(): array
    {
        $operators = parent::getDefaultOperatorInputByType();
        $operators['string'] = ['==', '!=', '{}', '!{}'];
        $operators['numeric'] = ['==', '!=', '{}', '!{}'];

        return $operators;
    }

    /**
     * Validate Address Rule Condition
     *
     * @param \Magento\Customer\Model\Customer|\Amasty\Segments\Model\GuestCustomerData $model
     * @return bool
     */
    public function validate(AbstractModel $model): bool
    {
        $customer = $this->objectValidation($model);
        if (!$customer) {
            return false;
        }

        $address = $this->getAddress($customer);

        if ($address instanceof AbstractModel) {
            return parent::validate($address);
        }

        if (!$address && !$model->getCustomerIsGuest()) {
            /**
             * If customer doesn't have default address, then validate all addresses.
             * If one of the all addresses will be valid, then customer is valid.
             */
            foreach ($model->getAddresses() as $address) {
                if (parent::validate($address)) {
                    return true;
                }
            }

            return false;
        }

        return $this->validateAttribute($address);
    }

    /**
     * Case and type insensitive comparison of values
     *
     * @param string|int|float $validatedValue
     * @param string|int|float $value
     * @param bool $strict
     * @return bool
     */
    protected function _compareValues($validatedValue, $value, $strict = true): bool
    {
        if ($strict && is_numeric($validatedValue) && is_numeric($value)) {
            return $validatedValue == $value;
        }

        return stripos($value, (string)$validatedValue) !== false;
    }

    protected function canValidateGuest(): bool
    {
        return true;
    }

    /**
     * @param \Magento\Customer\Model\Customer|\Amasty\Segments\Model\GuestCustomerData $model
     *
     * @return AbstractModel|bool
     */
    protected function getAddress($model)
    {
        switch ($this->type) {
            case 'billing':
                return $model->getDefaultBillingAddress();
            case 'shipping':
                return $model->getDefaultShippingAddress();
        }

        return $model;
    }
}
