<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/

namespace Amasty\Segments\Model\ResourceModel\Segment;

use Amasty\Segments\Api\Data\SegmentInterface;
use Amasty\Segments\Helper\Base as Base;
use Amasty\Segments\Model\ResourceModel\Segment as SegmentResource;
use Amasty\Segments\Model\Segment;
use Magento\Framework\Data\Collection\Db\FetchStrategyInterface;
use Magento\Framework\Data\Collection\EntityFactoryInterface;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;
use Psr\Log\LoggerInterface;

/**
 * @method Segment[] getItems()
 */
class Collection extends AbstractCollection
{
    /**
     * @var MetadataPool
     */
    private $metadataPool;

    public function __construct(
        EntityFactoryInterface $entityFactory,
        LoggerInterface $logger,
        FetchStrategyInterface $fetchStrategy,
        ManagerInterface $eventManager,
        MetadataPool $metadataPool,
        AdapterInterface $connection = null,
        AbstractDb $resource = null
    ) {
        parent::__construct(
            $entityFactory,
            $logger,
            $fetchStrategy,
            $eventManager,
            $connection,
            $resource
        );
        $this->metadataPool = $metadataPool;
    }

    protected function _construct()
    {
        $this->_init(Segment::class, SegmentResource::class);
        $this->_setIdFieldName($this->getResource()->getIdFieldName());
    }

    /**
     * @return $this
     */
    protected function _afterLoad()
    {
        $entityMetadata = $this->metadataPool->getMetadata(SegmentInterface::class);
        $this->performAfterLoad(
            'amasty_segments_website',
            'amasty_segments_entity_website',
            $entityMetadata->getLinkField(),
            'segment_id',
            'website_id'
        );

        return parent::_afterLoad();
    }

    /**
     * @return $this
     */
    public function addActiveFilter()
    {
        $this->addFieldToFilter(SegmentInterface::IS_ACTIVE, ['eq' => 1]);

        return $this;
    }

    /**
     * @param string $tableName
     * @param string|null $linkField
     * @return Collection
     */
    public function joinStoreRelationTable(
        $tableName = Base::AMASTY_SEGMENTS_WEBSITE_TABLE_NAME,
        $linkField = 'segment_id'
    ) {
        $this->getSelect()->join(
            ['store_table' => $this->getTable($tableName)],
            'main_table.' . SegmentInterface::SEGMENT_ID . ' = store_table.' . $linkField,
            ['website_id' => new \Zend_Db_Expr('GROUP_CONCAT(store_table.website_id)')]
        )->group('main_table.' . SegmentInterface::SEGMENT_ID);

        return $this;
    }

    /**
     * @param string $tableName
     * @param string $alias
     * @param string $linkField
     * @param string $fkField
     * @param string $targetField
     */
    protected function performAfterLoad($tableName, $alias, $linkField, $fkField, $targetField)
    {
        $linkedIds = $this->getColumnValues($linkField);

        if (count($linkedIds)) {
            $connection = $this->getConnection();
            $select = $connection->select()
                ->from([$alias => $this->getTable($tableName)])
                ->where($alias . '.' . $fkField . ' IN (?)', $linkedIds);
            $result = $connection->fetchAll($select);

            if ($result) {
                $data = [];

                foreach ($result as $item) {
                    $data[$item[$fkField]][] = $item[$targetField];
                }

                foreach ($this->getItems() as $item) {
                    $linkedId = $item->getData($linkField);
                    if (!isset($data[$linkedId])) {
                        continue;
                    }
                    $item->setData($targetField, $data[$linkedId]);
                }
            }
        }
    }

    /**
     * Perform operations before rendering filters
     * Join store relation table if there is store filter
     *
     * @return void
     */
    protected function _renderFiltersBefore()
    {
        if ($this->getFilter('website_id')) {
            $this->joinStoreRelationTable();
        }

        parent::_renderFiltersBefore();
    }
}
