<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/

namespace Amasty\Segments\Model\Providers;

use Amasty\Segments\Model\BatchLoader;
use Amasty\Segments\Model\GuestCustomerData;
use Amasty\Segments\Model\ResourceModel\Customer\Collection as CustomerCollection;
use Amasty\Segments\Model\ResourceModel\Customer\CollectionFactory as CustomerSegmentsCollectionFactory;
use Magento\Customer\Model\Customer;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Address;
use Magento\Store\Model\StoreManagerInterface;

class CustomerDataProvider
{
    public const COLLECTION_TYPE_QUOTE_FIELD = 'address_type';
    public const CUSTOMER_IS_GUEST_VALUE = 1;

    /**
     * @var \Magento\Customer\Model\ResourceModel\Customer\Collection
     */
    private $customerCollectionFactory;

    /**
     * @var \Amasty\Segments\Model\GuestCustomerDataFactory
     */
    private $guestCustomerDataFactory;

    /**
     * @var \Magento\Quote\Model\ResourceModel\Quote\CollectionFactory
     */
    private $quoteCollectionFactory;

    /**
     * @var \Amasty\Segments\Model\ResourceModel\Index
     */
    private $indexResource;

    /**
     * @var \Magento\Quote\Model\ResourceModel\Quote\Address\CollectionFactory
     */
    private $quoteAddressFactory;

    /**
     * @var CustomerSegmentsCollectionFactory
     */
    private $customerSegmentsCollectionFactory;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var array
     */
    private $webSiteToStoreMap = [];

    /**
     * @var BatchLoader
     */
    private $batchLoader;

    public function __construct(
        \Magento\Customer\Model\ResourceModel\Customer\CollectionFactory $customerCollectionFactory,
        \Amasty\Segments\Model\GuestCustomerDataFactory $guestCustomerDataFactory,
        \Magento\Quote\Model\ResourceModel\Quote\CollectionFactory $quoteCollectionFactory,
        \Amasty\Segments\Model\ResourceModel\Index $indexResource,
        \Magento\Quote\Model\ResourceModel\Quote\Address\CollectionFactory $quoteAddressFactory,
        CustomerSegmentsCollectionFactory $customerSegmentsCollectionFactory,
        StoreManagerInterface $storeManager,
        BatchLoader $batchLoader
    ) {
        $this->customerCollectionFactory = $customerCollectionFactory;
        $this->guestCustomerDataFactory = $guestCustomerDataFactory;
        $this->quoteCollectionFactory = $quoteCollectionFactory;
        $this->indexResource = $indexResource;
        $this->quoteAddressFactory = $quoteAddressFactory;
        $this->customerSegmentsCollectionFactory = $customerSegmentsCollectionFactory;
        $this->storeManager = $storeManager;
        $this->batchLoader = $batchLoader;
    }

    /**
     * @param $segment
     * @return \Generator
     */
    public function getGuestCustomers($segment): \Generator
    {
        $quoteCollection = $this->getGuestQuoteCollection();
        $customerCollection = $this->getCustomerCollection();
        $quoteCollection->addFieldToSelect('customer_email');
        $customerCollection->getSelect()
            ->reset('columns')
            ->columns('e.email')
            ->where('e.email IN (' . $quoteCollection->getSelect() .')');
        $websiteIds = $segment->getWebsiteId();

        if ($websiteIds) {
            $customerCollection->addFieldToFilter('website_id', ['in' => $websiteIds]);
        }

        $registeredEmails = [];

        foreach ($this->batchLoader->load($customerCollection) as $item) {
            $registeredEmails[] = $item->getEmail();
        }

        $quoteCollection
            ->addFieldToFilter(
                ['customer_email', 'customer_email'],
                [
                    ['null' => true],
                    ['nin'  => ($registeredEmails ?: 1)]
                ]
            );
        if ($websiteIds) {
            $quoteCollection->addFieldToFilter(
                'store_id',
                ['in' => $this->getStoreIdsByWebsiteIds($websiteIds)]
            );
        }
        $quoteCollection = $this->addGuestQuoteFilters($quoteCollection);

        /** @var Quote $quote */
        foreach ($this->batchLoader->load($quoteCollection) as $quote) {
            yield $this->prepareGuestCustomerModelFromQuote($quote);
        }
    }

    /**
     * @param $segment
     * @return \Generator
     */
    public function getCustomers($segment): \Generator
    {
        $customerCollection = $this->getCustomerSegmentsCollection();
        $websiteIds = $segment->getWebsiteId();

        if ($websiteIds) {
            $customerCollection->addFieldToFilter('website_id', ['in' => $websiteIds]);
        }

        /** @var Customer $customer */
        foreach ($this->batchLoader->load($customerCollection) as $customer) {
            yield $customer;
        }
    }

    /**
     * @param $segmentId
     * @return array
     */
    public function getQuoteIdsFromIndex($segmentId)
    {
        return $this->indexResource->getIdsFromIndex(
            CustomerCollection::AMASTY_SEGMENTS_INDEX_TABLE_QUOTE_FIELD_NAME,
            $segmentId
        );
    }

    /**
     * @param $collection
     * @return \Magento\Quote\Model\ResourceModel\Quote\Collection
     */
    private function addGuestQuoteFilters($collection)
    {
        return $collection
            ->addFieldToSelect('entity_id', 'quote_id')
            ->addFieldToSelect('store_id')
            ->addFieldToSelect('customer_email')
            ->addFieldToSelect('customer_is_guest')
            ->addFieldToSelect('created_at')
            ->addFieldToSelect('updated_at')
            ->addFieldToSelect('base_grand_total')
            ->addFieldToSelect('grand_total')
            ->addFieldToSelect('items_qty')
            ->addFieldToSelect('customer_group_id', 'group_id')
            ->addFieldToSelect('is_active');
    }

    /**
     * @param string $quoteId
     * @param string $type
     * @return array|null
     */
    private function getAddressByType($quoteId, $type)
    {
        $addressData = $this->quoteAddressFactory
            ->create()
            ->addFieldToSelect('firstname')
            ->addFieldToSelect('email')
            ->addFieldToSelect('lastname')
            ->addFieldToSelect('city')
            ->addFieldToSelect('region')
            ->addFieldToSelect('region_id')
            ->addFieldToSelect('country_id')
            ->addFieldToSelect('postcode')
            ->addFieldToSelect('telephone')
            ->addFieldToFilter(CustomerCollection::AMASTY_SEGMENTS_INDEX_TABLE_QUOTE_FIELD_NAME, ['eq' => $quoteId])
            ->addFieldToFilter(self::COLLECTION_TYPE_QUOTE_FIELD, ['eq' => $type])
            ->getFirstitem();

        return $addressData;
    }

    /**
     * @return \Magento\Quote\Model\ResourceModel\Quote\Collection
     */
    private function getGuestQuoteCollection()
    {
        return $this->quoteCollectionFactory
            ->create()
            ->addFieldToFilter(CustomerCollection::CUSTOMER_IS_GUEST_FIELD_NAME, ['eq' => 1]);
    }

    /**
     * @return \Magento\Customer\Model\ResourceModel\Customer\Collection
     */
    private function getCustomerCollection()
    {
        return $this->customerCollectionFactory->create();
    }

    /**
     * @return CustomerCollection
     */
    private function getCustomerSegmentsCollection()
    {
        return $this->customerSegmentsCollectionFactory->create();
    }

    /**
     * @param Quote $quote
     * @return GuestCustomerData
     */
    private function prepareGuestCustomerModelFromQuote(Quote $quote)
    {
        /** @var GuestCustomerData $questCustomer */
        $questCustomer = $this->guestCustomerDataFactory->create(
            ['data' => $quote->getData()]
        );
        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();
        if ($quoteId = $quote->getQuoteId()) {
            $shippingAddress = $this->getAddressByType($quoteId, Address::ADDRESS_TYPE_SHIPPING);
            $billingAddress = $this->getAddressByType($quoteId, Address::ADDRESS_TYPE_BILLING);
        }
        $questCustomer->setDefaultShippingAddress($shippingAddress);
        $questCustomer->setDefaultBillingAddress($billingAddress);

        return $questCustomer;
    }

    /**
     * @param array $websiteIds
     *
     * @return array
     */
    private function getStoreIdsByWebsiteIds(array $websiteIds): array
    {
        if (!$this->webSiteToStoreMap) {
            $stores = $this->storeManager->getStores();

            /** @var \Magento\Store\Api\Data\StoreInterface $store */
            foreach ($stores as $store) {
                $this->webSiteToStoreMap[$store->getWebsiteId()][] = $store->getId();
            }
        }

        return array_values(array_intersect_key($this->webSiteToStoreMap, array_flip($websiteIds)));
    }
}
