<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/

namespace Amasty\Segments\Model\Indexer;

use Amasty\Segments\Api\Data\SegmentInterface;
use Amasty\Segments\Helper\Base;
use Amasty\Segments\Helper\Order\Data;
use Amasty\Segments\Model\Providers\CustomerDataProvider;
use Amasty\Segments\Model\ResourceModel\Customer\Collection as CustomerCollection;
use Amasty\Segments\Model\ResourceModel\Index;
use Amasty\Segments\Model\ResourceModel\Segment\CollectionFactory as SegmentCollectionFactory;
use Amasty\Segments\Model\Segment;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\ResourceModel\Db\VersionControl\Snapshot;
use Psr\Log\LoggerInterface;

class IndexBuilder
{
    /**
     * @var ResourceConnection
     */
    protected $resource;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var AdapterInterface
     */
    protected $connection;

    /**
     * @var SegmentCollectionFactory
     */
    protected $segmentCollectionFactory;

    /**
     * @var IndexerQueue
     */
    protected $indexerQueue;

    /**
     * @var Index
     */
    protected $indexResource;

    /**
     * @var CustomerDataProvider
     */
    private $customerDataProvider;

    /**
     * @var array
     */
    private $insertBatch = [];

    /**
     * @var Data
     */
    private $helper;

    /**
     * @var Snapshot
     */
    private $snapshot;

    public function __construct(
        SegmentCollectionFactory $segmentCollectionFactory,
        ResourceConnection $resource,
        LoggerInterface $logger,
        IndexerQueue $indexerQueue,
        Index $indexResource,
        CustomerDataProvider $customerDataProvider,
        Data $helper,
        Snapshot $snapshot
    ) {
        $this->resource = $resource;
        $this->connection = $resource->getConnection();
        $this->logger = $logger;
        $this->segmentCollectionFactory = $segmentCollectionFactory;
        $this->indexerQueue = $indexerQueue;
        $this->indexResource = $indexResource;
        $this->customerDataProvider = $customerDataProvider;
        $this->helper = $helper;
        $this->snapshot = $snapshot;
    }

    /**
     * @throws LocalizedException
     */
    public function reindexByQueue()
    {
        try {
            $this->doReindexByQueue();
        } catch (\Exception $e) {
            $this->critical($e);
            throw new LocalizedException(__($e->getMessage()), $e);
        }
    }

    /**
     * @return $this
     * @throws LocalizedException
     */
    protected function doReindexByQueue()
    {
        $segmentIdx = $this->getSegmentIdsForIndexByEvents();

        if (count($segmentIdx) > 0) {
            $this->reindexByIds($segmentIdx);
            $this->cleanEventTable();
        }

        return $this;
    }

    /**
     * Reindex by ids
     *
     * @param array $ids
     * @throws LocalizedException
     * @return void
     * @api
     */
    public function reindexByIds(array $ids)
    {
        try {
            $this->doReindexByIds($ids);
        } catch (\Exception $e) {
            $this->critical($e);
            throw new LocalizedException(__('Amasty segments indexing failed. See details in exception log.'));
        }
    }

    /**
     * @param $id
     * @throws LocalizedException
     */
    public function reindexById($id)
    {
        try {
            $this->doReindexByIds($id);
        } catch (\Exception $e) {
            $this->critical($e);
            throw new LocalizedException(__('Amasty segments indexing failed. See details in exception log.'));
        }
    }

    /**
     * @param $ids
     * @return $this
     */
    protected function doReindexByIds($ids)
    {
        /** @var \Amasty\Segments\Model\ResourceModel\Segment\Collection $segmentCollection */
        $segmentCollection = $this->getSegmentCollection()
            ->addFieldToFilter(SegmentInterface::SEGMENT_ID, ['in' => $ids]);

        if ($segmentCollection->getSize()) {
            $this->indexResource->cleanBySegmentIds($ids);
            $this->segmentIndexByCollection($segmentCollection);
        }

        return $this;
    }

    /**
     * Full reindex
     *
     * @throws LocalizedException
     * @return void
     * @api
     */
    public function reindexFull()
    {
        try {
            $this->doReindexFull();
        } catch (\Exception $e) {
            $this->critical($e);
            throw new LocalizedException(__($e->getMessage()), $e);
        }
    }

    /**
     * @param $segmentCollection
     * @throws \Exception
     */
    protected function segmentIndexByCollection($segmentCollection)
    {
        /** @var Segment $segment */
        foreach ($segmentCollection as $segment) {
            foreach ($this->customerDataProvider->getGuestCustomers($segment) as $guestCustomer) {
                $segmentIndexData = $this->precessCustomer($segment, $guestCustomer);
                $this->batchInsert($segmentIndexData);
            }

            foreach ($this->customerDataProvider->getCustomers($segment) as $customer) {
                $segmentIndexData = $this->precessCustomer($segment, $customer);
                $this->batchInsert($segmentIndexData);
            }
        }

        if (!empty($this->insertBatch)) {
            $this->indexResource->insertIndexData($this->insertBatch);
        }
    }

    private function batchInsert(?array $segmentIndexData)
    {
        if ($segmentIndexData === null) {
            return;
        }

        if (count($this->insertBatch) >= $this->helper->getBatchSize()) {
            $this->indexResource->insertIndexData($this->insertBatch);
            $this->insertBatch = [];
        } else {
            $this->insertBatch[] = $segmentIndexData;
        }
    }

    private function precessCustomer(Segment $segment, $customer): ?array
    {
        $result = null;
        $isValid = false;

        if ($rule = $segment->getSalesRule()) {
            $isValid = $rule->validate($customer);

            if (method_exists($this->snapshot, 'clear')) {
                $this->snapshot->clear();
            }
        }

        if ($isValid) {
            $segmentIndexData = [
                'customer_id' => null,
                'quote_id' => null,
                SegmentInterface::SEGMENT_ID => $segment->getSegmentId()
            ];

            if ($customer->getCustomerIsGuest()) {
                $segmentIndexData['quote_id'] = $customer->getQuoteId();
            } else {
                $segmentIndexData['customer_id'] = $customer->getId();
            }

            $result = $segmentIndexData;
        }

        return $result;
    }

    /**
     * Full reindex Template method
     *
     * @throws \Exception
     * @return void
     */
    protected function doReindexFull()
    {
        $this->indexResource->cleanAllIndex();
        /** @var \Amasty\Segments\Model\ResourceModel\Segment\Collection $segmentCollection */
        $segmentCollection = $this->getSegmentCollection();
        $this->segmentIndexByCollection($segmentCollection);
    }

    /**
     * @return $this
     */
    protected function cleanEventTable()
    {
        $this->indexerQueue->cleanAll();

        return $this;
    }

    /**
     * @return SegmentCollectionFactory
     */
    protected function getSegmentCollection()
    {
        return $this->segmentCollectionFactory->create()->addActiveFilter();
    }

    /**
     * @return array
     */
    protected function getSegmentIdsForIndexByEvents()
    {
        $query = $this->connection->select()
            ->from(
                $this->resource->getTableName(Base::AMASTY_SEGMENTS_EVENT_TABLE_NAME),
                SegmentInterface::SEGMENT_ID
            )->distinct();

        return $this->connection->fetchAll($query);
    }

    /**
     * @param \Exception $e
     * @return void
     */
    protected function critical($e)
    {
        $this->logger->critical($e);
    }

    /**
     * @param $segmentIds
     * @param $field
     * @param $fieldId
     * @return $this
     */
    public function insertBySegmentIds($segmentIds, $field, $fieldId)
    {
        $insertData = [];

        foreach ($segmentIds as $segmentId) {
            $addedFields = $this->getAddedFieldsByField($field, $fieldId);
            $addedFields[SegmentInterface::SEGMENT_ID] = $segmentId;
            $insertData[] = $addedFields;
        }

        $this->indexResource->insertIndexData($insertData);

        return $this;
    }

    /**
     * @param $field
     * @param $fieldValue
     * @return mixed
     */
    private function getAddedFieldsByField($field, $fieldValue)
    {
        $res = [
            CustomerCollection::AMASTY_SEGMENTS_INDEX_TABLE_CUSTOMER_FIELD_NAME => null,
            CustomerCollection::AMASTY_SEGMENTS_INDEX_TABLE_QUOTE_FIELD_NAME => null
        ];

        if (in_array($field, array_keys($res))) {
            $res[$field] = $fieldValue;
        }

        return $res;
    }
}
