<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Customer Segmentation for Magento 2
*/

namespace Amasty\Segments\Helper;

use Magento\Framework\App\Helper\AbstractHelper;

class Base extends AbstractHelper
{
    /**
     * Defining all module constants
     **/
    public const CONFIG_PATH_GENERAL_CRON_TIME = 'segments/general/frequency';
    public const CONFIG_PATH_GENERAL_CUSTOMER_ATTRIBUTES = 'segments/general/customer_attributes';
    public const CONFIG_PATH_GENERAL_ORDER_ATTRIBUTES = 'segments/general/order_attributes';
    public const CONFIG_PATH_GENERAL_BATCH_SIZE = 'segments/general/batch_size';
    public const CURRENT_SEGMENT_REGISTRY_NAMESPACE = 'current_amasty_segments_segment';
    public const CURRENT_SEGMENT_CUSTOMER_COLLECTION_REGISTRY = 'current_amasty_segments_segment_customer_collection';

    public const AMASTY_SEGMENTS_WEBSITE_TABLE_NAME = 'amasty_segments_website';
    public const AMASTY_SEGMENTS_EVENT_TABLE_NAME = 'amasty_segments_event';
    public const AMASTY_SEGMENTS_SEGMENT_TABLE_NAME = 'amasty_segments_segment';
    public const AMASTY_SEGMENTS_INDEX_TABLE_NAME = 'amasty_segments_index';

    public const ARRAY_FILTER_LIKE = 'like';
    public const ARRAY_FILTER_EQUALS = 'eq';
    public const ARRAY_FILTER_LESS_THAN_OR_EQUALS = 'lteq';
    public const ARRAY_FILTER_GRATER_THAN_OR_EQUALS = 'gteq';
    public const ARRAY_FILTER_IN = 'in';

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface|null
     */
    protected $_scopeConfig = null;

    /**
     * @var \Magento\Framework\Registry
     */
    protected $coreRegistry;

    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    protected $date;

    /**
     * Base constructor.
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\Registry $coreRegistry
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\Registry $coreRegistry,
        \Magento\Framework\Stdlib\DateTime\DateTime $date
    ) {
        parent::__construct($context);
        $this->_scopeConfig = $context->getScopeConfig();
        $this->coreRegistry = $coreRegistry;
        $this->date = $date;
    }

    /**
     * @param $path
     * @param null $storeId
     * @param string $scope
     * @return mixed
     */
    public function getConfigValueByPath(
        $path,
        $storeId = null,
        $scope = \Magento\Store\Model\ScopeInterface::SCOPE_STORE
    ) {
        return $this->_scopeConfig->getValue($path, $scope, $storeId);
    }

    /**
     * @param int $days
     * @param string $format
     * @return string
     */
    public function getDateDiffFormat($days, $format = 'Y-m-d')
    {
        $newDiffDate = $this->date->gmtTimestamp() - (int)$days * 60 * 60 * 24;

        return $this->date->gmtDate($format, $newDiffDate);
    }

    public function getDateWithPositiveOffset(int $days, string $format): string
    {
        $newDiffDate = $this->date->gmtTimestamp() + $days * 86400;

        return $this->date->gmtDate($format, $newDiffDate);
    }

    /**
     * @param int    $days
     * @param string $format
     *
     * @return int
     */
    public function getTimestampDiffFormat($days, $format = 'Y-m-d')
    {
        $newDiffDate = $this->date->gmtTimestamp($this->date->gmtDate($format)) - (int)$days * 60 * 60 * 24;

        return $newDiffDate;
    }

    /**
     * @param string $date
     * @param string $format
     *
     * @return int
     */
    public function convertToTimestamp($date, $format = 'Y-m-d')
    {
        return $this->date->gmtTimestamp($this->getFormatDate($date, $format));
    }

    /**
     * @param string $date
     * @param string $format
     * @return string
     */
    public function getFormatDate($date, $format = 'Y-m-d')
    {
        return $this->date->gmtDate($format, $date);
    }

    /**
     * @param string $format
     * @return false|string
     */
    public function getCurrentDate($format = 'Y-m-d')
    {
        return $this->date->gmtDate($format, $this->date->gmtTimestamp());
    }

    /**
     * @param $segment
     * @return mixed
     */
    public function initCurrentSegment($segment)
    {
        $this->coreRegistry->register(
            self::CURRENT_SEGMENT_REGISTRY_NAMESPACE,
            $segment
        );

        return $segment;
    }

    /**
     * @param $segmentConditions
     * @param $classPath
     * @return bool
     */
    public function checkExistConditionInSegment($segmentConditions, $classPath)
    {
        $res = false;

        foreach ($segmentConditions as $condition) {
            if (strpos($condition->getType(), $classPath) !== false) {
                $res = true;

                break;
            }
        }

        return $res;
    }

    public function getBatchSize(): int
    {
        return (int)$this->_scopeConfig->getValue(self::CONFIG_PATH_GENERAL_BATCH_SIZE);
    }
}
