/**
 * Login Attribute input
 */
define([
    'jquery',
    'underscore',
    'Magento_Ui/js/form/element/abstract',
    'Amasty_CustomerLoginAttributes/js/model/login-attributes-model',
    'Magento_Customer/js/customer-data',
    'mage/translate'
], function ($, _, Abstract, loginAttributesModel, customerData, $t) {
    'use strict';

    return Abstract.extend({
        defaults: {
            isRequired: false,
            skipDelayedValidation: false,
            validationInputDelay: 1000,
            validationResponse: null,
            isDuplicate: false,
            duplicateMark: false,
            isValid: true,
            storageKey: 'amasty-login-attributes',
            selectors: {
                amCheckoutLoginForm: '.amcheckout-form-login',
                amCheckoutPasswordField: '#customer-password'
            }
        },

        initialize: function () {
            this._super();

            this.isFocused.subscribe(function (isFocused) {
                var value = this.value();

                if (isFocused || _.isEmpty(value)) {
                    return;
                }

                this.validateAndSaveToStorage(value);
                this.bubble('attribute-update');
            }.bind(this));

            this.setValueFromStorage();

            return this;
        },

        initObservable: function () {
            this._super().observe('isFocused');

            return this;
        },

        validate: function () {
            var result = this._super(),
                response = this.validationResponse,
                responseJSON,
                errorMessage,
                $element;

            if (!result.valid || this.disabled() || !this.visible()) {
                this.isValid = result.valid;

                return result;
            }

            if (!_.isEmpty(response)) {
                responseJSON = response.responseJSON;
                errorMessage = responseJSON.message.replace('%1', responseJSON.parameters);
                result.valid = false;
            }

            if (this.isDuplicate) {
                errorMessage = $t('"%1" is used more than once'.replace('%1', this.value()));
                result.valid = false;
            }

            this.throwErrorMessage(errorMessage);

            if (this.source && !this.isValid) {
                this.source.set('params.invalid', true);
            }

            $element = $('#' + this.uid);

            if ($element.length) {
                $element.valid();
            }

            this.isValid = result.valid;

            return result;
        },

        reset: function () {
            this.value(this.initialValue);

            if (this.isValid) { // fixed hiding errors after Paypal validation reset
                this.error(false);
            }

            return this;
        },

        /**
         * @param {String} message
         * @returns {void}
         */
        throwErrorMessage: function (message) {
            this.error(message);
            this.error.valueHasMutated();
            this.bubble('error', message);
        },

        getAttributeCode: function () {
            return this.attributeCode;
        },

        /**
         * @param {String} value
         * @returns {void}
         */
        validateAndSaveToStorage: function (value) {
            var $loginForm,
                $customerPassword;

            if (this.skipDelayedValidation) {
                this.skipDelayedValidation = false;

                return;
            }

            loginAttributesModel.validate(this, this.getAttributeCode(), value);
            this.saveValueToStorage(value);

            // trigger Amasty OSC validateAndSaveRegistration action
            $loginForm = $(this.selectors.amCheckoutLoginForm);
            $customerPassword = $(this.selectors.amCheckoutPasswordField);

            if ($loginForm.length && $customerPassword.val() && $loginForm.valid()) {
                this.bubble('save-login-form');
                this.bubble('save-for-cart');
            }
        },

        /**
         * @param {String} value
         * @returns {void}
         */
        saveValueToStorage: function (value) {
            var code = this.getAttributeCode(),
                attributesData;

            attributesData = customerData.get(this.storageKey)();
            attributesData[code] = value;
            customerData.set(this.storageKey, attributesData);
        },

        setValueFromStorage: function () {
            var code = this.getAttributeCode(),
                attributesData = customerData.get(this.storageKey)();

            if (_.isEmpty(attributesData) || _.isEmpty(attributesData[code])) {
                return;
            }

            this.value(attributesData[code]);
            this.validateAndSaveToStorage(attributesData[code]);
            this.skipDelayedValidation = true;
        }
    });
});
