<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\ViewModel\Order;

use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeInterface;
use Amasty\CustomerLoginAttributes\Api\ValueGetListInterface;
use Amasty\CustomerLoginAttributes\Model\ResourceModel\LoginAttribute\CollectionFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Sales\Api\OrderRepositoryInterface;

class AttributesForOrderView implements ArgumentInterface
{
    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;

    /**
     * @var CollectionFactory
     */
    private $attributeCollectionFactory;

    /**
     * @var ValueGetListInterface
     */
    private $valueGetList;

    public function __construct(
        OrderRepositoryInterface $orderRepository,
        CollectionFactory $attributeCollectionFactory,
        ValueGetListInterface $valueGetList
    ) {
        $this->orderRepository = $orderRepository;
        $this->attributeCollectionFactory = $attributeCollectionFactory;
        $this->valueGetList = $valueGetList;
    }

    /**
     * @param int $orderId
     * @return array Array of Array: array(array('label' => '...', 'value' => '...'))
     */
    public function getCustomerAttributeValuesFromOrder(int $orderId): array
    {
        if (!$orderId) {
            return [];
        }

        try {
            $order = $this->orderRepository->get($orderId);
        } catch (NoSuchEntityException $e) {
            return [];
        }

        $customerId = (int)$order->getCustomerId();

        if (!$customerId) {
            return [];
        }

        $attributesCollection = $this->attributeCollectionFactory->create();
        $attributesCollection->addFieldToFilter(LoginAttributeInterface::IS_SHOW_ON_ORDER_VIEW, 1);

        $values = $this->valueGetList->getByCustomerId($customerId);
        $valuesArray = [];

        foreach ($values as $value) {
            $valuesArray[$value->getAttributeCode()] = $value->getValue();
        }

        $data = [];
        /** @var LoginAttributeInterface $item */
        foreach ($attributesCollection->getItems() as $item) {
            $value = $valuesArray[$item->getAttributeCode()] ?? '';
            if (!$value) {
                continue;
            }
            $data[] = [
                'label' => $item->getLabel(),
                'value' => $value
            ];
        }

        return $data;
    }
}
