<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\ViewModel;

use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeCustomerValueInterface;
use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeInterface;
use Amasty\CustomerLoginAttributes\Api\ValueGetListInterface;
use Amasty\CustomerLoginAttributes\Model\ConfigProvider;
use Amasty\CustomerLoginAttributes\Model\LoginAttribute\CollectionRegistry;
use Amasty\CustomerLoginAttributes\Model\LoginAttribute\LabelProvider;
use Amasty\CustomerLoginAttributes\Model\ResourceModel\LoginAttribute\Collection;
use Amasty\CustomerLoginAttributes\Plugin\AdminOutput\Customer\Ui\Component\Form\AddLoginAttributesToCustomerFormPlugin;
use Magento\Customer\Model\Session;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Store\Model\StoreManagerInterface;

class LoginAttributes implements ArgumentInterface
{
    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var LabelProvider
     */
    private $labelProvider;

    /**
     * @var string
     */
    private $fieldArea;

    /**
     * @var string|null
     */
    private $conditionField;

    /**
     * @var Session
     */
    private $customerSession;

    /**
     * @var LoginAttributeCustomerValueInterface[]|null
     */
    private $values = null;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var ValueGetListInterface
     */
    private $valueGetList;

    /**
     * @var CollectionRegistry
     */
    private $collectionRegistry;

    public function __construct(
        StoreManagerInterface $storeManager,
        LabelProvider $labelProvider,
        Session $customerSession,
        ConfigProvider $configProvider,
        ValueGetListInterface $valueGetList,
        CollectionRegistry $collectionRegistry,
        string $fieldArea = AddLoginAttributesToCustomerFormPlugin::FIELD_AREA,
        ?string $conditionField = null
    ) {
        $this->storeManager = $storeManager;
        $this->labelProvider = $labelProvider;
        $this->fieldArea = $fieldArea;
        $this->conditionField = $conditionField;
        $this->customerSession = $customerSession;
        $this->configProvider = $configProvider;
        $this->valueGetList = $valueGetList;
        $this->collectionRegistry = $collectionRegistry;
    }

    /**
     * @return LoginAttributeInterface[]
     */
    public function getLoginAttributes(): array
    {
        return $this->getAttributesCollection()->getItems();
    }

    /**
     * @param LoginAttributeInterface $loginAttribute
     * @return string
     */
    public function getLabelForAttribute(LoginAttributeInterface $loginAttribute): string
    {
        $storeId = (int)$this->storeManager->getStore()->getId();

        return $this->labelProvider->getLabelForStore($loginAttribute, $storeId);
    }

    /**
     * @param LoginAttributeInterface $loginAttribute
     * @param array $overrideValues
     * @return string
     */
    public function getValueForAttribute(LoginAttributeInterface $loginAttribute, array $overrideValues = []): string
    {
        if (isset($overrideValues[$this->fieldArea][$loginAttribute->getAttributeCode()])) {
            return $overrideValues[$this->fieldArea][$loginAttribute->getAttributeCode()];
        }

        return $this->getCustomerValues()[$loginAttribute->getAttributeCode()] ?? '';
    }

    /**
     * @return string
     */
    public function getFieldArea(): string
    {
        return $this->fieldArea;
    }

    /**
     * @return LoginAttributeCustomerValueInterface[]
     */
    private function getCustomerValues(): array
    {
        if ($this->values === null) {
            $this->values = [];
            $customerId = (int)$this->customerSession->getCustomerId();

            if (!$customerId) {
                return $this->values;
            }
            $values = $this->valueGetList->getByCustomerId($customerId);

            foreach ($values as $value) {
                $this->values[$value->getAttributeCode()] = $value->getValue();
            }
        }

        return $this->values;
    }

    /**
     * @return Collection
     */
    private function getAttributesCollection(): Collection
    {
        $storeId = (int)$this->storeManager->getStore()->getId();

        return $this->collectionRegistry->getCollectionFor($this->conditionField)->addFilterByStore($storeId);
    }

    /**
     * @return bool
     */
    public function isAttributesExists(): bool
    {
        return $this->getAttributesCollection()->getSize() > 0;
    }

    /**
     * Get login attributes' fields notice
     * @return string
     */
    public function getAttributeFieldNotice(): string
    {
        if ($this->configProvider->isAttributeNotice()) {
            return $this->configProvider->getAttributeNotice();
        }

        return '';
    }
}
