<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Test\Unit\ViewModel;

use Amasty\CustomerLoginAttributes\Model\ConfigProvider;
use Amasty\CustomerLoginAttributes\Model\EnabledValidator;
use Amasty\CustomerLoginAttributes\ViewModel\LoginViewModel;

class LoginViewModelTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var ConfigProvider|\PHPUnit\Framework\MockObject\MockObject
     */
    private $configProviderMock;

    /**
     * @var EnabledValidator|\PHPUnit\Framework\MockObject\MockObject
     */
    private $enabledValidatorMock;

    /**
     * @var LoginViewModel
     */
    private $model;

    public function setUp(): void
    {
        $this->configProviderMock = $this->createMock(ConfigProvider::class);
        $this->enabledValidatorMock = $this->createMock(EnabledValidator::class);

        $this->model = new LoginViewModel($this->configProviderMock, $this->enabledValidatorMock);
    }

    /**
     * @param bool $isEnabled
     * @param bool $expectedResult
     * @covers LoginViewModel::isEnabled()
     * @dataProvider isEnabledDataProvider
     */
    public function testIsEnabled(bool $isEnabled, bool $expectedResult): void
    {
        $this->enabledValidatorMock->method('isEnabled')->willReturn($isEnabled);

        $this->assertSame($expectedResult, $this->model->isEnabled());
    }

    /**
     * @param bool $isOverride
     * @param string $newTitle
     * @param bool $expectedResult
     * @covers LoginViewModel::isOverrideLoginTitle()
     * @dataProvider isOverrideTitleDataProvider
     */
    public function testIsOverrideLoginTitle(bool $isOverride, string $newTitle, bool $expectedResult): void
    {
        $this->configProviderMock->method('isNewLoginFieldTitle')->willReturn($isOverride);
        $this->configProviderMock->method('getNewLoginFieldTitle')->willReturn($newTitle);

        $result = $this->model->isOverrideLoginTitle();

        $this->assertSame($expectedResult, $result);
    }

    /**
     * @param bool $isOverride
     * @param string $newNotice
     * @param bool $expectedResult
     * @covers LoginViewModel::isOverrideLoginNotice()
     * @dataProvider isOverrideNoticeDataProvider
     */
    public function testIsOverrideLoginNotice(bool $isOverride, string $newNotice, bool $expectedResult): void
    {
        $this->configProviderMock->method('isNewLoginNotice')->willReturn($isOverride);
        $this->configProviderMock->method('getNewLoginNotice')->willReturn($newNotice);

        $result = $this->model->isOverrideLoginNotice();

        $this->assertSame($expectedResult, $result);
    }

    /**
     * @return array
     */
    public function isEnabledDataProvider(): array
    {
        return [
            'true' => [true, true],
            'false' => [false, false],
        ];
    }

    /**
     * @return array
     */
    public function isOverrideTitleDataProvider(): array
    {
        return [
            'disabled' => [
                'isOverride' => false,
                'newTitle' => 'NEW TITLE',
                'expectedResult' => false
            ],
            'enabledEmpty' => [
                'isOverride' => false,
                'newTitle' => '',
                'expectedResult' => false
            ],
            'enabled' => [
                'isOverride' => true,
                'newTitle' => 'NEW TITLE',
                'expectedResult' => true
            ],
        ];
    }

    /**
     * @return array
     */
    public function isOverrideNoticeDataProvider(): array
    {
        return [
            'disabled' => [
                'isOverride' => false,
                'newNotice' => 'NEW Notice',
                'expectedResult' => false
            ],
            'enabledEmpty' => [
                'isOverride' => false,
                'newNotice' => '',
                'expectedResult' => false
            ],
            'enabled' => [
                'isOverride' => true,
                'newNotice' => 'NEW Notice',
                'expectedResult' => true
            ],
        ];
    }
}
