<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Test\Unit\Model\LoginAttribute;

use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeInterface;
use Amasty\CustomerLoginAttributes\Api\LoginAttributeGetInterface;
use Amasty\CustomerLoginAttributes\Model\LoginAttribute\Validator;
use Amasty\CustomerLoginAttributes\Model\Source\Option\LoginAttributes;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;

class ValidatorTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var LoginAttributeGetInterface|\PHPUnit\Framework\MockObject\MockObject
     */
    private $loginAttributeGetInterfaceMock;

    /**
     * @var Validator
     */
    private $model;

    public function setUp(): void
    {
        $this->loginAttributeGetInterfaceMock = $this->createMock(LoginAttributeGetInterface::class);

        $this->model = new Validator($this->loginAttributeGetInterfaceMock);
    }

    /**
     * @param string $attributeCode
     * @param int $attributeId
     * @param string $existAttributeCode
     * @param int $existAttributeId
     * @throws InputException
     * @covers Validator::validate()
     * @dataProvider exceptionDataProvider
     */
    public function testValidateException(
        string $attributeCode,
        int $attributeId,
        string $existAttributeCode,
        int $existAttributeId
    ): void {
        $this->expectException(InputException::class);

        $loginAttributeMock = $this->createMock(LoginAttributeInterface::class);
        $loginAttributeMock->method('getAttributeCode')->willReturn($attributeCode);
        $loginAttributeMock->method('getAttributeId')->willReturn($attributeId);

        if ($existAttributeCode) {
            $existAttributeMock = $this->createMock(LoginAttributeInterface::class);
            $existAttributeMock->method('getAttributeCode')->willReturn($existAttributeCode);
            $existAttributeMock->method('getAttributeId')->willReturn($existAttributeId);

            $this->loginAttributeGetInterfaceMock->method('getByCode')->willReturn($existAttributeMock);
        } else {
            $this->loginAttributeGetInterfaceMock
                ->method('getByCode')
                ->willThrowException(new NoSuchEntityException(__('Test')));
        }

        $this->model->validate($loginAttributeMock);
    }

    /**
     * @throws InputException
     * @covers Validator::validate()
     */
    public function testValidateSuccess(): void
    {
        $loginAttributeMock = $this->createMock(LoginAttributeInterface::class);
        $loginAttributeMock->method('getAttributeCode')->willReturn('attr1');
        $loginAttributeMock->method('getAttributeId')->willReturn(1);

        $this->loginAttributeGetInterfaceMock
            ->method('getByCode')
            ->willThrowException(new NoSuchEntityException(__('Test')));

        $this->model->validate($loginAttributeMock);
    }

    /**
     * @throws InputException
     * @covers Validator::validate()
     */
    public function testValidateSuccessNoChanges(): void
    {
        $loginAttributeMock = $this->createMock(LoginAttributeInterface::class);
        $loginAttributeMock->method('getAttributeCode')->willReturn('attr1');
        $loginAttributeMock->method('getAttributeId')->willReturn(1);

        $this->loginAttributeGetInterfaceMock
            ->method('getByCode')
            ->willReturn($loginAttributeMock);

        $this->model->validate($loginAttributeMock);
    }

    /**
     * @return array[]
     */
    public function exceptionDataProvider(): array
    {
        return [
            'incorrectAttributeCode' => [
                'attributeCode' => ' efwef fwe wef',
                'attributeId' => 1,
                'existAttributeCode' => '',
                'existAttributeId' => 0,
            ],
            'notAllowedAttributeCode' => [
                'attributeCode' => LoginAttributes::EMAIL,
                'attributeId' => 1,
                'existAttributeCode' => '',
                'existAttributeId' => 0,
            ],
            'duplicatedAttributeCode' => [
                'attributeCode' => 'attr1',
                'attributeId' => 1,
                'existAttributeCode' => 'attr1',
                'existAttributeId' => 2,
            ],
            'changeAttributeCode' => [
                'attributeCode' => 'attr1',
                'attributeId' => 1,
                'existAttributeCode' => 'attr2',
                'existAttributeId' => 1,
            ],
        ];
    }
}
