<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Test\Unit\Model;

use Amasty\CustomerLoginAttributes\Model\ConfigProvider;
use Amasty\CustomerLoginAttributes\Model\EnabledValidator;
use Amasty\CustomerLoginAttributes\Model\Source\Option\LoginAttributes;
use PHPUnit\Framework\MockObject\MockObject;

class EnabledValidatorTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var EnabledValidator
     */
    private $model;

    /**
     * @var ConfigProvider|MockObject
     */
    private $configProviderMock;

    public function setUp(): void
    {
        $this->configProviderMock = $this->createMock(ConfigProvider::class);

        $this->model = new EnabledValidator($this->configProviderMock);
    }

    /**
     * @param array $listOfAttributes
     * @param bool $expectedResult
     *
     * @covers \Amasty\CustomerLoginAttributes\Model\EnabledValidator::isEnabled
     * @dataProvider dataProvider
     */
    public function testIsEnabled(array $listOfAttributes, bool $expectedResult): void
    {
        $this->configProviderMock
            ->expects($this->once())
            ->method('getLoginAttributes')
            ->willReturn($listOfAttributes);

        $actualResult = $this->model->isEnabled();

        $this->assertSame($expectedResult, $actualResult);
    }

    /**
     * @param array $listOfAttributes
     * @param bool $expectedResult
     *
     * @covers \Amasty\CustomerLoginAttributes\Model\EnabledValidator::isEmailLoginEnabled
     * @dataProvider dataProviderEmailEnabled
     */
    public function testIsEmailLoginEnabled(array $listOfAttributes, bool $expectedResult): void
    {
        $this->configProviderMock
            ->expects($this->once())
            ->method('getLoginAttributes')
            ->willReturn($listOfAttributes);

        $actualResult = $this->model->isEmailLoginEnabled();

        $this->assertSame($expectedResult, $actualResult);
    }

    /**
     * @return array
     */
    public function dataProvider(): array
    {
        return [
            'onlyLoginAttributes' => [
                'listOfAttributes' => [
                    'attributeA',
                    'attributeB',
                    'attributeC',
                ],
                'expectedResult' => true,
            ],
            'emailAndLoginAttributes' => [
                'listOfAttributes' => [
                    LoginAttributes::EMAIL,
                    'attributeA',
                    'attributeB',
                    'attributeC',
                ],
                'expectedResult' => true,
            ],
            'onlyEmail' => [
                'listOfAttributes' => [
                    LoginAttributes::EMAIL
                ],
                'expectedResult' => false,
            ]
        ];
    }

    /**
     * @return array
     */
    public function dataProviderEmailEnabled(): array
    {
        return [
            'onlyLoginAttributes' => [
                'listOfAttributes' => [
                    'attributeA',
                    'attributeB',
                    'attributeC',
                ],
                'expectedResult' => false,
            ],
            'emailAndLoginAttributes' => [
                'listOfAttributes' => [
                    LoginAttributes::EMAIL,
                    'attributeA',
                    'attributeB',
                    'attributeC',
                ],
                'expectedResult' => true,
            ],
            'onlyEmail' => [
                'listOfAttributes' => [
                    LoginAttributes::EMAIL
                ],
                'expectedResult' => true,
            ]
        ];
    }
}
