<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Plugin\AdminOutput\Order\Ui\Component\Listing;

use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeCustomerValueInterface;
use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeInterface;
use Amasty\CustomerLoginAttributes\Model\ResourceModel\LoginAttribute\CollectionFactory;
use Amasty\CustomerLoginAttributes\Model\ResourceModel\LoginAttributeCustomerValue\CollectionFactory
    as ValuesCollectionFactory;
use Amasty\CustomerLoginAttributes\Ui\Component\Listing\LoginAttributesColumns;
use Magento\Framework\View\Element\UiComponent\DataProvider\DataProvider;
use Magento\Framework\Api\Search\SearchCriteria;
use Magento\Framework\Api\Search\SearchResultInterface;

class AddLoginAttributesDataPlugin
{
    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var string
     */
    private $columnPrefix;

    /**
     * @var ValuesCollectionFactory
     */
    private $valuesCollectionFactory;

    public function __construct(
        CollectionFactory $collectionFactory,
        ValuesCollectionFactory $valuesCollectionFactory,
        string $columnPrefix = LoginAttributesColumns::GRID_ATTRIBUTE_COLUMN_PREFIX
    ) {
        $this->collectionFactory = $collectionFactory;
        $this->columnPrefix = $columnPrefix;
        $this->valuesCollectionFactory = $valuesCollectionFactory;
    }

    /**
     * Workaround for get data with split database feature enabled. Remove after split database feature removal
     *
     * @param DataProvider $subject
     * @param array $result
     * @return array
     */
    public function afterGetData(DataProvider $subject, array $result): array
    {
        if (!$this->allowToAddAttributes($subject)) {
            return $result;
        }

        $customerIds = [];
        foreach ($result['items'] as $key => $item) {
            $customerId = $item['customer_id'];
            if (!isset($customerIds[$customerId])) {
                $customerIds[$customerId] = [];
            }
            $customerIds[$customerId][] = $key;
        }

        $valuesCollection = $this->valuesCollectionFactory->create();
        $valuesCollection->addFieldToFilter(
            LoginAttributeCustomerValueInterface::CUSTOMER_ID,
            ['in' => array_keys($customerIds)]
        );
        $valuesCollection->addAttributesToSelect();
        $valuesCollection->addFieldToFilter(LoginAttributeInterface::IS_SHOW_ON_ORDER_GRID, 1);

        foreach ($valuesCollection as $value) {
            $attributeCode = $value->getData(LoginAttributeInterface::ATTRIBUTE_CODE);
            $itemsKeys = $customerIds[$value->getCustomerId()];

            foreach ($itemsKeys as $itemKey) {
                $result['items'][$itemKey][$this->columnPrefix . $attributeCode] = $value->getValue();
            }
        }

        return $result;
    }

    /**
     * Modify search criteria for filters working
     * @param DataProvider $subject
     * @param SearchCriteria $searchCriteria
     * @return SearchCriteria
     */
    public function afterGetSearchCriteria(DataProvider $subject, SearchCriteria $searchCriteria): SearchCriteria
    {
        // @TODO: remove true after split database feature removal
        if (!$this->allowToAddAttributes($subject) || true) {
            return $searchCriteria;
        }

        foreach ($searchCriteria->getFilterGroups() as $filterGroup) {
            foreach ($filterGroup->getFilters() as $filter) {
                if (strpos($filter->getField(), $this->columnPrefix) === 0) {
                    $filter->setField($filter->getField() . '.value');
                }
            }
        }

        return $searchCriteria;
    }

    /**
     * Add data to collection
     * @param DataProvider $subject
     * @param SearchResultInterface $collection
     * @return SearchResultInterface
     */
    public function afterGetSearchResult(
        DataProvider $subject,
        SearchResultInterface $collection
    ): SearchResultInterface {

        // @TODO: remove true after split database feature removal
        if (!$this->allowToAddAttributes($subject) || true) {
            return $collection;
        }
        $attributesCollection = $this->collectionFactory->create();
        $attributesCollection->addFieldToFilter(LoginAttributeInterface::IS_SHOW_ON_ORDER_GRID, 1);

        $attributeValuesTable = $attributesCollection->getTable('amasty_customer_login_attribute_customer_value');
        /** @var LoginAttributeInterface $attribute */
        foreach ($attributesCollection->getItems() as $attribute) {
            $alias = $this->columnPrefix . $attribute->getAttributeCode();
            $collection->getSelect()
                ->joinLeft(
                    [$alias => $attributeValuesTable],
                    "{$alias}.customer_id = main_table.customer_id AND {$alias}.attribute_id = "
                    . $attribute->getAttributeId(),
                    [$alias => $alias . '.value']
                );
        }

        return $collection;
    }

    /**
     * Is can add Customer Login Attributes' Columns Data to Component
     *
     * @param DataProvider $dataProvider
     *
     * @return bool
     */
    public function allowToAddAttributes(DataProvider $dataProvider): bool
    {
        return $dataProvider->getName() === 'sales_order_grid_data_source';
    }
}
