<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Plugin\AdminOutput\Customer\Ui\Component\Form;

use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeInterface;
use Amasty\CustomerLoginAttributes\Api\ValueGetListInterface;
use Amasty\CustomerLoginAttributes\Model\ResourceModel\LoginAttribute\CollectionFactory;
use Magento\Customer\Model\Customer\DataProviderWithDefaultAddresses;
use Magento\Ui\Component\Form\Field;

class AddLoginAttributesToCustomerFormPlugin
{
    public const FIELD_AREA = 'am_login_attributes';

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var string
     */
    private $fieldArea;

    /**
     * @var ValueGetListInterface
     */
    private $valueGetList;

    public function __construct(
        CollectionFactory $collectionFactory,
        ValueGetListInterface $valueGetList,
        string $fieldArea = self::FIELD_AREA
    ) {
        $this->collectionFactory = $collectionFactory;
        $this->fieldArea = $fieldArea;
        $this->valueGetList = $valueGetList;
    }

    /**
     * Add customer login attributes to form data
     *
     * @param DataProviderWithDefaultAddresses $subject
     * @param array $result
     * @return array
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterGetData(DataProviderWithDefaultAddresses $subject, array $result): array
    {
        foreach ($result as $customerId => &$customerData) {
            if (!$customerId) {
                continue;
            }
            $attributeValues = $this->valueGetList->getByCustomerId((int)$customerId);
            foreach ($attributeValues as $attributeValue) {
                $fieldName = $attributeValue->getAttributeCode();
                $customerData['customer'][$this->fieldArea][$fieldName] = $attributeValue->getValue();
            }
        }

        return $result;
    }

    /**
     * Add customer login attributes to form fields
     *
     * @param DataProviderWithDefaultAddresses $subject
     * @param array $result
     * @return array
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterGetMeta(DataProviderWithDefaultAddresses $subject, array $result): array
    {
        $attributes = $this->collectionFactory->create()->getItems();

        /** @var LoginAttributeInterface $attribute */
        foreach ($attributes as $attribute) {
            $fieldCode = $this->fieldArea . '.' . $attribute->getAttributeCode();
            $validationRules = [];
            if ($attribute->isRequired()) {
                $validationRules['required-entry'] = true;
            }
            if ($attribute->getInputValidation()) {
                $validationRules[$attribute->getInputValidation()] = true;
            }

            $result['customer']['children'][$fieldCode]['arguments']['data']['config'] = [
                'componentType' => Field::NAME,
                'dataType' => 'text',
                'formElement' => 'input',
                'visible' => true,
                'required' => $attribute->isRequired(),
                'label' => $attribute->getLabel(),
                'sortOrder' => $attribute->getSortOrder(),
                'validation' => $validationRules,
                'placeholder' => $attribute->getPlaceholder()
            ];
        }

        return $result;
    }
}
