<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Model\Wrapper;

/**
 * Wrapper for libphonenumber library
 * @see \libphonenumber\PhoneNumberUtil
 */
class PhoneLibWrapper
{
    /**
     * Cast phone number to international format
     * @param string $phoneNumber
     * @param string|null $countryCode
     * @return string|null
     */
    public function getInternationalNumber(string $phoneNumber, ?string $countryCode): ?string
    {
        if (!$this->isLibraryInstalled()) {
            return $phoneNumber;
        }

        $phoneUtil = \libphonenumber\PhoneNumberUtil::getInstance();
        try {
            $phoneNumberProto = $phoneUtil->parse($phoneNumber, $countryCode);

            return $phoneUtil->format($phoneNumberProto, \libphonenumber\PhoneNumberFormat::E164);
        } catch (\libphonenumber\NumberParseException $e) {
            return null;
        }
    }

    /**
     * @param string $phoneNumber
     * @param string|null $countryCode
     * @return bool
     */
    public function isValid(string $phoneNumber, ?string $countryCode): bool
    {
        if (!$this->isLibraryInstalled()) {
            return true;
        }

        $phoneUtil = \libphonenumber\PhoneNumberUtil::getInstance();
        try {
            $phoneNumberProto = $phoneUtil->parse($phoneNumber, $countryCode);
        } catch (\libphonenumber\NumberParseException $e) {
            return false;
        }

        return $phoneUtil->isValidNumber($phoneNumberProto);
    }

    /**
     * @return bool
     */
    private function isLibraryInstalled(): bool
    {
        return class_exists(\libphonenumber\PhoneNumberUtil::class);
    }
}
