<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Model\ResourceModel;

use Magento\Framework\DataObject;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;

class LoginAttribute extends AbstractDb
{
    public const TABLE = 'amasty_customer_login_attribute';
    public const STORES_TABLE = 'amasty_customer_login_attribute_store';

    /**
     * @var array
     */
    protected $_serializableFields = [
        'store_labels' => [
            '{}',
            []
        ]
    ];

    public function _construct()
    {
        $this->_init(self::TABLE, 'attribute_id');
    }

    /**
     * @param AbstractModel $model
     * @param array $storeIds
     * @return AbstractModel
     * @throws \Magento\Framework\Exception\AlreadyExistsException
     * @throws \Exception
     */
    public function saveWithStores(AbstractModel $model, array $storeIds): AbstractModel
    {
        $connection = $this->getConnection();
        $connection->beginTransaction();
        try {
            $this->save($model);
            $this->saveStores((int)$model->getId(), $storeIds);
        } catch (\Exception $e) {
            $connection->rollBack();
            throw $e;
        }
        $connection->commit();

        return $model;
    }

    /**
     * Update stores table according to new store ids list
     *
     * @param int $attributeId
     * @param int[] $storeIds
     */
    public function saveStores(int $attributeId, array $storeIds): void
    {
        $connection = $this->getConnection();
        $storesTable = $this->getTable(self::STORES_TABLE);
        $connection->delete($storesTable, ['attribute_id = ?' => $attributeId]);

        $insertArray = [];

        foreach ($storeIds as $storeId) {
            $insertArray[] = [
                'attribute_id' => $attributeId,
                'store_id' => $storeId
            ];
        }

        if ($insertArray) {
            $connection->insertMultiple($storesTable, $insertArray);
        }
    }

    /**
     * @param int $attributeId
     * @return array list of store ids for attribute
     */
    public function loadStores(int $attributeId): array
    {
        $select = $this->getConnection()->select()
            ->from($this->getTable(self::STORES_TABLE), ['store_id'])
            ->where('attribute_id = ?', $attributeId);

        return $this->getConnection()->fetchCol($select);
    }

    /**
     * Unserialize serializeable object fields
     *
     * Workaround for deserializing \Magento\Framework\View\Element\UiComponent\DataProvider\Document::class
     *
     * @param DataObject $object
     * @return void
     */
    public function unserializeFieldsInDataObject(DataObject $object): void
    {
        foreach ($this->_serializableFields as $field => $parameters) {
            list(, $unserializeDefault) = $parameters;
            $this->_unserializeField($object, $field, $unserializeDefault);
        }
    }
}
