<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Model\LoginAttributeCustomerValue;

use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeCustomerValueInterface;
use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeCustomerValueInterfaceFactory;
use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeInterface;
use Amasty\CustomerLoginAttributes\Api\ValueDeleteInterface;
use Amasty\CustomerLoginAttributes\Api\ValueSaveInterface;
use Amasty\CustomerLoginAttributes\Model\ResourceModel\LoginAttribute\CollectionFactory;
use Amasty\CustomerLoginAttributes\Model\ResourceModel\LoginAttributeCustomerValue\CollectionFactory
    as ValueCollectionFactory;

class CustomerValuesUpdater
{
    /**
     * @var CollectionFactory
     */
    private $attributeCollectionFactory;

    /**
     * @var LoginAttributeCustomerValueInterfaceFactory
     */
    private $valueFactory;

    /**
     * @var ValueSaveInterface
     */
    private $valueSave;

    /**
     * @var ValueDeleteInterface
     */
    private $valueDelete;

    /**
     * @var ValueCollectionFactory
     */
    private $valueCollectionFactory;

    public function __construct(
        CollectionFactory $attributeCollectionFactory,
        ValueCollectionFactory $valueCollectionFactory,
        LoginAttributeCustomerValueInterfaceFactory $valueFactory,
        ValueSaveInterface $valueSave,
        ValueDeleteInterface $valueDelete
    ) {
        $this->attributeCollectionFactory = $attributeCollectionFactory;
        $this->valueFactory = $valueFactory;
        $this->valueSave = $valueSave;
        $this->valueDelete = $valueDelete;
        $this->valueCollectionFactory = $valueCollectionFactory;
    }

    /**
     * Update customer login attributes according to array of values
     * @param int $customerId
     * @param array $attributesData
     */
    public function updateCustomerAttributes(int $customerId, array $attributesData): void
    {
        $newAttributeValues = $this->updateExistingAttributes($customerId, $attributesData);
        $attributesData = array_filter($newAttributeValues);

        if (!$attributesData) {
            return;
        }
        $this->createNewAttributes($customerId, $attributesData);
    }

    /**
     * @param int $customerId
     * @param array $attributesData
     * @return array List of new attributes for customer
     */
    private function updateExistingAttributes(int $customerId, array $attributesData): array
    {
        $oldCustomerValues = $this->valueCollectionFactory->create();
        $oldCustomerValues->addAttributesToSelect();
        $oldCustomerValues->addFieldToFilter(LoginAttributeCustomerValueInterface::CUSTOMER_ID, $customerId);

        foreach ($oldCustomerValues->getItems() as $attributeValue) {
            $attributeCode = $attributeValue->getData(LoginAttributeInterface::ATTRIBUTE_CODE);
            if (!array_key_exists($attributeCode, $attributesData)) {
                continue;
            }

            $oldValue = $attributeValue->getValue();
            $newValue = $attributesData[$attributeCode];

            // remove existingDataFromList
            unset($attributesData[$attributeCode]);
            if ($oldValue === $newValue) {
                continue;
            }

            if (!$newValue) {
                $this->valueDelete->delete($attributeValue);
            } else {
                $attributeValue->setValue($newValue);
                $this->valueSave->save($attributeValue);
            }
        }

        return $attributesData;
    }

    /**
     * @param int $customerId
     * @param array $attributesData
     */
    private function createNewAttributes(int $customerId, array $attributesData): void
    {
        $attributesCollection = $this->attributeCollectionFactory->create();
        $attributesCollection->addFieldToFilter(
            LoginAttributeInterface::ATTRIBUTE_CODE,
            ['in' => array_keys($attributesData)]
        );

        /** @var LoginAttributeInterface $attribute */
        foreach ($attributesCollection->getItems() as $attribute) {
            $value = $attributesData[$attribute->getAttributeCode()];
            $valueModel = $this->valueFactory->create();
            $valueModel->setAttributeId($attribute->getAttributeId());
            $valueModel->setCustomerId($customerId);
            $valueModel->setValue($value);
            $this->valueSave->save($valueModel);
        }
    }
}
