<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Model\LoginAttribute;

use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeInterface;
use Amasty\CustomerLoginAttributes\Api\LoginAttributeGetInterface;
use Amasty\CustomerLoginAttributes\Model\Source\Option\LoginAttributes;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;

class Validator
{
    public const REG_EXP = '/^[a-z]+[a-z0-9_]+$/';
    public const NOT_ALLOWED_CODES = [LoginAttributes::EMAIL];

    /**
     * @var LoginAttributeGetInterface
     */
    private $loginAttributeGet;

    /**
     * @var array
     */
    private $notAllowedCodes;

    /**
     * @var string
     */
    private $regExp;

    public function __construct(
        LoginAttributeGetInterface $loginAttributeGet,
        array $notAllowedCodes = self::NOT_ALLOWED_CODES,
        string $regExp = self::REG_EXP
    ) {
        $this->loginAttributeGet = $loginAttributeGet;
        $this->notAllowedCodes = $notAllowedCodes;
        $this->regExp = $regExp;
    }

    /**
     * @param LoginAttributeInterface $loginAttribute
     * @throws InputException
     */
    public function validate(LoginAttributeInterface $loginAttribute): void
    {
        $attributeCode = $loginAttribute->getAttributeCode();
        if (!preg_match($this->regExp, $attributeCode)) {
            throw new InputException(__('Incorrect attribute code'));
        }

        if (in_array($attributeCode, $this->notAllowedCodes, true)) {
            throw new InputException(__('Attribute code %1 is not allowed', $attributeCode));
        }

        try {
            $existAttribute = $this->loginAttributeGet->getByCode($attributeCode);
        } catch (NoSuchEntityException $e) {
            $existAttribute = null;
        }

        if ($existAttribute && $existAttribute->getAttributeId() !== $loginAttribute->getAttributeId()) {
            throw new InputException(__('Attribute code must be unique!'));
        }

        if ($existAttribute && $loginAttribute->getAttributeCode() !== $existAttribute->getAttributeCode()) {
            throw new InputException(__('Change of attribute_code is not allowed!'));
        }
    }
}
