<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Controller\Customer;

use Amasty\CustomerLoginAttributes\Api\ValueBulkSaveFromCustomerInterface;
use Amasty\CustomerLoginAttributes\Model\LoginAttributeCustomerValue\Converter\ArrayToObjectsConverter;
use Amasty\CustomerLoginAttributes\Plugin\AdminOutput\Customer\Ui\Component\Form\AddLoginAttributesToCustomerFormPlugin;
use Magento\Customer\Model\Session;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\Result\RedirectFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Message\ManagerInterface as MessageManagerInterface;

class EditPost implements HttpPostActionInterface, CsrfAwareActionInterface
{

    /**
     * @var Session
     */
    private $customerSession;

    /**
     * @var RedirectFactory
     */
    private $redirectFactory;

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var MessageManagerInterface
     */
    private $messageManager;

    /**
     * @var ValueBulkSaveFromCustomerInterface
     */
    private $valueBulkSaveFromCustomer;

    /**
     * @var ArrayToObjectsConverter
     */
    private $arrayToObjectsConverter;

    public function __construct(
        Session $customerSession,
        RedirectFactory $redirectFactory,
        RequestInterface $request,
        MessageManagerInterface $messageManager,
        ValueBulkSaveFromCustomerInterface $valueBulkSaveFromCustomer,
        ArrayToObjectsConverter $arrayToObjectsConverter
    ) {
        $this->customerSession = $customerSession;
        $this->redirectFactory = $redirectFactory;
        $this->request = $request;
        $this->messageManager = $messageManager;
        $this->valueBulkSaveFromCustomer = $valueBulkSaveFromCustomer;
        $this->arrayToObjectsConverter = $arrayToObjectsConverter;
    }

    /**
     * @return \Magento\Framework\Controller\Result\Redirect
     */
    public function execute()
    {
        $customerId = (int)$this->customerSession->getCustomerId();
        $redirect = $this->redirectFactory->create();
        $redirect->setPath('customer/account/edit');
        if (!$customerId) {
            return $redirect;
        }
        $attributeValues = $this->request->getParam(AddLoginAttributesToCustomerFormPlugin::FIELD_AREA);
        if (!is_array($attributeValues)) {
            $attributeValues = [];
        }
        $attributeValues = $this->arrayToObjectsConverter->convert($attributeValues);

        $password = (string)$this->request->getParam('current_password');

        try {
            $this->valueBulkSaveFromCustomer->save($customerId, $attributeValues, $password);
            $this->messageManager->addSuccessMessage(__('Values successfully saved'));
        } catch (\Magento\Framework\Validator\Exception $e) {
            foreach ($e->getMessages() as $error) {
                $this->messageManager->addErrorMessage($error->getText());
            }
        } catch (LocalizedException $e) {
            $this->messageManager->addErrorMessage($e->getMessage());
        } catch (\Exception $e) {
            $this->messageManager->addExceptionMessage($e, __('Something went wrong with saving values'));
        }

        return $redirect;
    }

    /**
     * Create exception in case CSRF validation failed.
     * Return null if default exception will suffice.
     *
     * @param RequestInterface $request
     *
     * @return InvalidRequestException|null
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        $resultRedirect = $this->redirectFactory->create();
        $resultRedirect->setPath('customer/account/edit');

        return new InvalidRequestException(
            $resultRedirect,
            [__('Invalid Form Key. Please refresh the page.')]
        );
    }

    /**
     * Perform custom request validation.
     * Return null if default validation is needed.
     *
     * @param RequestInterface $request
     *
     * @return bool|null
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return null;
    }
}
