<?php

declare(strict_types=1);

namespace Amasty\CustomerLoginAttributes\Block\Onepage;

use Amasty\CustomerLoginAttributes\Api\Data\LoginAttributeInterface;
use Amasty\CustomerLoginAttributes\Model\LoginAttribute\CollectionRegistry;
use Amasty\CustomerLoginAttributes\Model\LoginAttribute\LabelProvider;
use Magento\Checkout\Block\Checkout\LayoutProcessorInterface;
use Magento\Checkout\Model\Session;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Store\Model\StoreManagerInterface;

class LoginAttributes implements LayoutProcessorInterface
{
    /**
     * @var CollectionRegistry
     */
    private $collectionRegistry;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var LabelProvider
     */
    private $labelProvider;

    /**
     * @var ArrayManager
     */
    private $arrayManager;

    /**
     * @var Session
     */
    private $checkoutSession;

    public function __construct(
        CollectionRegistry $collectionRegistry,
        StoreManagerInterface $storeManager,
        LabelProvider $labelProvider,
        ArrayManager $arrayManager,
        Session $checkoutSession
    ) {
        $this->collectionRegistry = $collectionRegistry;
        $this->storeManager = $storeManager;
        $this->labelProvider = $labelProvider;
        $this->arrayManager = $arrayManager;
        $this->checkoutSession = $checkoutSession;
    }

    /**
     * @param array $jsLayout
     * @return array
     */
    public function process($jsLayout): array
    {
        $attributesComponent = $this->getCheckoutLoginAttributes();

        if (!$attributesComponent) {
            return $jsLayout;
        }

        $isQuoteVirtual = $this->checkoutSession->getQuote()->isVirtual();

        $delimiter = '.';
        $basePath = 'components.checkout.children.steps.children.';
        $shipping = 'shipping-step.children.shippingAddress.children.customer-email.children.login-attributes.children';
        $billing = 'billing-step.children.payment.children.customer-email.children.login-attributes';

        if ($this->arrayManager->exists($basePath . $shipping, $jsLayout, $delimiter)) {
            $jsLayout = $this->arrayManager->merge($basePath . $shipping, $jsLayout, $attributesComponent, $delimiter);
        }

        if (!$isQuoteVirtual) {
            $jsLayout = $this->arrayManager->remove($basePath . $billing, $jsLayout, $delimiter);
        } elseif ($this->arrayManager->exists($basePath . $billing . '.children', $jsLayout, $delimiter)) {
            $jsLayout = $this->arrayManager->merge(
                $basePath . $billing . '.children',
                $jsLayout,
                $attributesComponent,
                $delimiter
            );
        }

        return $jsLayout;
    }

    /**
     * @return array
     */
    private function getCheckoutLoginAttributes(): array
    {
        $storeId = (int)$this->storeManager->getStore()->getId();
        $attributesCollection = $this->collectionRegistry->getCollectionFor(
            LoginAttributeInterface::IS_SHOW_ON_AMASTY_CHECKOUT
        )->addFilterByStore($storeId);

        $attributesComponent = [];
        /** @var LoginAttributeInterface $attribute */
        foreach ($attributesCollection->getItems() as $attribute) {
            $validation = [];
            if ($attribute->isRequired()) {
                $validation['required-entry'] = true;
            }

            if ($attribute->getInputValidation()) {
                $validation[$attribute->getInputValidation()] = true;
            }

            $attributeCode = $attribute->getAttributeCode();

            $attributesComponent[$attributeCode] = [
                'component' => 'Amasty_CustomerLoginAttributes/js/view/form/element/login-attribute-input',
                'config' => [
                    'provider' => 'checkoutProvider',
                    'deps' => [
                        'checkoutProvider'
                    ],
                    'template' => 'ui/form/field',
                    'elementTmpl' => 'Amasty_CustomerLoginAttributes/form/element/input'
                ],
                'dataScope' => $attributeCode,
                'attributeCode' => $attributeCode,
                'label' => $this->labelProvider->getLabelForStore($attribute, $storeId),
                'placeholder' => $attribute->getPlaceholder(),
                'validation' => $validation,
                'isRequired' => $attribute->isRequired()
            ];
        }

        return $attributesComponent;
    }
}
