<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Login as Customer for Magento 2
*/

declare(strict_types=1);

namespace Amasty\CustomerLogin\Ui\Component\Listing\Column;

use Amasty\CustomerLogin\Model\ConfigProvider;
use Magento\Customer\Model\ResourceModel\CustomerRepository;
use Magento\Customer\Ui\Component\Listing\Column\Actions;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\UiComponent\ContextInterface;
use Magento\Framework\View\Element\UiComponentFactory;
use Magento\Store\Model\StoreManagerInterface;

class GridActions extends Actions
{
    /**
     * @var \Magento\Customer\Model\ResourceModel\CustomerRepository
     */
    private $customerRepository;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        ContextInterface $context,
        UiComponentFactory $uiComponentFactory,
        UrlInterface $urlBuilder,
        CustomerRepository $customerRepository,
        ConfigProvider $configProvider,
        StoreManagerInterface $storeManager,
        ScopeConfigInterface $scopeConfig,
        array $components = [],
        array $data = []
    ) {
        parent::__construct($context, $uiComponentFactory, $urlBuilder, $components, $data);
        $this->customerRepository = $customerRepository;
        $this->storeManager = $storeManager;
        $this->scopeConfig = $scopeConfig;
        $this->configProvider = $configProvider;
    }

    /**
     * @param array $dataSource
     *
     * @return array
     */
    public function prepareDataSource(array $dataSource): array
    {
        $dataSource = parent::prepareDataSource($dataSource);

        if (!isset($dataSource['data']['items'])) {
            return $dataSource;
        }

        $target = $this->configProvider->isNewTab() ? '_blank' : '_self';
        $allowSelectStore = $this->configProvider->isAllowSelectStore();
        $customerAccountShare = $this->scopeConfig->getValue('customer/account_share/scope');

        foreach ($dataSource['data']['items'] as &$item) {
            if ($customerAccountShare) {
                if (!$allowSelectStore) {
                    $item[$this->getData('name')]['login'] = [
                        'href' => $this->getLoginLink((int)$item['entity_id'], 0, 0),
                        'target' => $target,
                        'label' => __('Login as Customer')
                    ];
                } else {
                    $customer = $this->customerRepository->getById($item['entity_id']);
                    $customerWebsiteId = (int)$customer->getWebsiteId();
                    $website = $this->storeManager->getWebsite($customerWebsiteId);

                    foreach ($website->getStores() as $storeId => $store) {
                        $item[$this->getData('name')]['login_' . $storeId] = [
                            'href' => $this->getLoginLink((int)$item['entity_id'], $customerWebsiteId, (int)$storeId),
                            'target' => $target,
                            'label' => __('Login on `%1` (%2)', $store->getName(), $website->getName())
                        ];
                    }
                }
            } else {
                if (!$allowSelectStore) {
                    foreach ($this->storeManager->getWebsites() as $websiteId => $website) {
                        $item[$this->getData('name')]['login_' . $websiteId] = [
                            'href' => $this->getLoginLink((int)$item['entity_id'], (int)$websiteId, 0),
                            'target' => $target,
                            'label' => __('Login on Website `%1`', $website->getName())
                        ];
                    }
                } else {
                    foreach ($this->storeManager->getWebsites() as $websiteId => $website) {
                        foreach ($website->getStores() as $storeId => $store) {
                            $item[$this->getData('name')]['login_' . $websiteId . '_' . $storeId] = [
                                'href' => $this->getLoginLink((int)$item['entity_id'], (int)$websiteId, (int)$storeId),
                                'target' => $target,
                                'label' => __('Login on `%1` (%2)', $store->getName(), $website->getName())
                            ];
                        }
                    }
                }
            }
        }

        return $dataSource;
    }

    /**
     * @param int $customerId
     * @param int $website
     * @param int $storeId
     *
     * @return string
     */
    private function getLoginLink(int $customerId, int $websiteId, int $storeId): string
    {
        $data = ['customer_id' => $customerId];

        if ($websiteId) {
            $data['website'] = (int)$websiteId;
        }

        if ($storeId) {
            $data['store'] = $storeId;
        }

        return $this->urlBuilder->getUrl('amcustomerlogin/customer/login', $data);
    }
}
