<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Login as Customer for Magento 2
*/

declare(strict_types=1);

namespace Amasty\CustomerLogin\Controller\Index;

use Amasty\CustomerLogin\Model\LoggedInRepository;
use Amasty\CustomerLogin\Model\ResourceModel\LoggedIn\CollectionFactory;
use Magento\Customer\Model\Session;
use Magento\Customer\Model\Url as CustomerUrl;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Stdlib\Cookie\CookieMetadataFactory;
use Magento\Framework\Stdlib\CookieManagerInterface;
use Magento\Framework\Url;
use Magento\Store\Api\StoreCookieManagerInterface;
use Magento\Store\Model\StoreManagerInterface;

class Index extends Action
{
    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var LoggedInRepository
     */
    private $repository;

    /**
     * @var Session
     */
    private $customerSession;

    /**
     * @var CustomerUrl
     */
    private $customerUrl;

    /**
     * @var CookieManagerInterface
     */
    private $cookieManager;

    /**
     * @var CookieMetadataFactory
     */
    private $cookieMetadataFactory;

    /**
     * @var Url
     */
    private $url;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var StoreCookieManagerInterface
     */
    private $storeCookieManager;

    public function __construct(
        Context $context,
        CollectionFactory $collectionFactory,
        CookieManagerInterface $cookieManager,
        CookieMetadataFactory $cookieMetadataFactory,
        LoggedInRepository $repository,
        Session $customerSession,
        CustomerUrl $customerUrl,
        Url $url,
        StoreManagerInterface $storeManager,
        StoreCookieManagerInterface $storeCookieManager
    ) {
        parent::__construct($context);
        $this->collectionFactory = $collectionFactory;
        $this->repository = $repository;
        $this->customerSession = $customerSession;
        $this->customerUrl = $customerUrl;
        $this->cookieManager = $cookieManager;
        $this->cookieMetadataFactory = $cookieMetadataFactory;
        $this->url = $url;
        $this->storeManager = $storeManager;
        $this->storeCookieManager = $storeCookieManager;
    }

    /**
     * @return Redirect
     */
    public function execute(): Redirect
    {
        if ($token = $this->getRequest()->getParam('token')) {
            /** @var \Amasty\CustomerLogin\Model\ResourceModel\LoggedIn\Collection $collection */
            $collection = $this->collectionFactory->create();
            $collection->addFieldToFilter('main_table.secret_key', $token)
                ->setPageSize(1)
                ->setCurPage(1);
            if ($collection->count()) {
                /** @var \Amasty\CustomerLogin\Model\LoggedIn $loggedIn */
                $loggedIn = $collection->getFirstItem();
                $previousCustomerId = $this->customerSession->getCustomerId();
                $storeId = $loggedIn->getStoreId();

                try {
                    if ($previousCustomerId !== null && $previousCustomerId != $loggedIn->getCustomerId()) {
                        $this->customerSession->logout();
                        $this->prepareStoreCookie($storeId);

                        return $this->resultRedirectFactory->create()->setUrl(
                            $this->url->setScope($storeId)->getUrl(
                                'amcustomerlogin/index/index',
                                ['token' => $token, '_nosid' => true]
                            )
                        );
                    }

                    $loggedIn->setSecretKey('');
                    $this->repository->save($loggedIn);
                    $this->customerSession->loginById($loggedIn->getCustomerId());
                    $this->customerSession->setAmCustomerLoginAdminId($loggedIn->getAdminId());
                    $this->prepareStoreCookie($storeId);
                    $this->flushSections(['customer', 'cart']);

                    return $this->resultRedirectFactory->create()->setUrl(
                        $this->customerUrl->getDashboardUrl()
                    );
                } catch (\Exception $e) {
                    null;
                }
            }
        }

        return $this->resultRedirectFactory->create()->setPath('/');
    }

    private function prepareStoreCookie(int $storeId)
    {
        $store = $this->storeManager->getStore($storeId);
        $this->storeCookieManager->setStoreCookie($store);

        if ($this->cookieManager->getCookie('mage-cache-sessid')) {
            $metadata = $this->cookieMetadataFactory->createCookieMetadata();
            $metadata->setPath('/');
            $this->cookieManager->deleteCookie('mage-cache-sessid', $metadata);
        }
    }

    private function flushSections(array $sectionNames): void
    {
        $metadata = $this->cookieMetadataFactory
            ->createPublicCookieMetadata()
            ->setPath('/');

        $sectionDataIds = json_decode($this->cookieManager->getCookie('section_data_ids', '{}'), true);

        foreach ($sectionNames as $sectionName) {
            $sectionDataIds[$sectionName] = isset($sectionDataIds[$sectionName]) ?
                $sectionDataIds[$sectionName] + 1000 :
                1000;
        }

        $this->cookieManager->deleteCookie('section_data_ids');
        $this->cookieManager->setPublicCookie(
            'section_data_ids',
            json_encode($sectionDataIds),
            $metadata
        );
    }
}
