<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Login as Customer for Magento 2
*/

declare(strict_types=1);

namespace Amasty\CustomerLogin\Controller\Adminhtml\Customer;

use Amasty\CustomerLogin\Model\ConfigProvider;
use Amasty\CustomerLogin\Model\LoggedInFactory;
use Amasty\CustomerLogin\Model\LoggedInRepository;
use Magento\Backend\App\Action;
use Magento\Backend\Model\Auth\Session;
use Magento\Customer\Model\ResourceModel\CustomerRepository;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Url;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\Math\Random;

class Login extends Action
{
    public const ADMIN_RESOURCE = 'Amasty_CustomerLogin::admin_login';

    /**
     * @var CustomerRepository
     */
    private $customerRepository;

    /**
     * @var Session
     */
    private $adminSession;

    /**
     * @var LoggedInFactory
     */
    private $loggedInFactory;

    /**
     * @var LoggedInRepository
     */
    private $loggedInRepository;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var Url
     */
    private $url;

    /**
     * @var Random
     */
    private $random;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        Action\Context $context,
        CustomerRepository $customerRepository,
        Session $adminSession,
        LoggedInFactory $loggedInFactory,
        LoggedInRepository $loggedInRepository,
        StoreManagerInterface $storeManager,
        Url $url,
        Random $random,
        ConfigProvider $configProvider
    ) {
        parent::__construct($context);
        $this->customerRepository = $customerRepository;
        $this->adminSession = $adminSession;
        $this->loggedInFactory = $loggedInFactory;
        $this->loggedInRepository = $loggedInRepository;
        $this->storeManager = $storeManager;
        $this->url = $url;
        $this->random = $random;
        $this->configProvider = $configProvider;
    }

    /**
     * @return Redirect
     */
    public function execute(): Redirect
    {
        if ($customerId = (int)$this->getRequest()->getParam('customer_id')) {
            try {
                /** @var \Amasty\CustomerLogin\Model\LoggedIn $loggedIn */
                $loggedIn = $this->loggedInFactory->create();
                $customer = $this->customerRepository->getById($customerId);
                $adminUser = $this->adminSession->getUser();
                $websiteId = $this->getRequest()->getParam('website');
                $storeId = $this->getRequest()->getParam('store');

                if ($websiteId === null || !($website = $this->storeManager->getWebsite($websiteId))) {
                    $website = $this->storeManager->getWebsite($customer->getWebsiteId());
                }

                if (!$this->configProvider->isAllowSelectStore()) {
                    $storeId = $customer->getStoreId();

                    if ($storeId !== null) {
                        $websiteIdByCustomerStoreId = $this->storeManager->getStore($storeId)->getWebsiteId();

                        if ($website->getId() != $websiteIdByCustomerStoreId) {
                            $storeId = $website->getDefaultStore()->getId();
                        }
                    } else {
                        $storeId = $website->getDefaultStore()->getId();
                    }
                }

                $secretKey = $this->random->getUniqueHash();
                $loggedIn->setCustomerId((int)$customer->getId())
                    ->setCustomerEmail($customer->getEmail())
                    ->setCustomerName($customer->getFirstname())
                    ->setCustomerLastName($customer->getLastname())
                    ->setAdminId((int)$adminUser->getId())
                    ->setAdminEmail($adminUser->getEmail())
                    ->setAdminUsername($adminUser->getUserName())
                    ->setWebsiteId((int)$website->getId())
                    ->setWebsiteCode($website->getCode())
                    ->setSecretKey($secretKey)
                    ->setStoreId((int)$storeId);
                $this->loggedInRepository->save($loggedIn);

                return $this->resultRedirectFactory->create()->setUrl(
                    $this->url->setScope($storeId)->getUrl(
                        'amcustomerlogin/index/index',
                        ['token' => $secretKey, '_nosid' => true]
                    )
                );
            } catch (\Exception $e) {
                $this->messageManager->addErrorMessage(__('Something went wrong.'));
            }
        }

        return $this->resultRedirectFactory->create()->setRefererUrl();
    }
}
