<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Login as Customer for Magento 2
*/

declare(strict_types=1);

namespace Amasty\CustomerLogin\Block\Adminhtml\Edit;

use Amasty\CustomerLogin\Model\ConfigProvider;
use Magento\Backend\Block\Widget\Button\SplitButton;
use Magento\Backend\Block\Widget\Context;
use Magento\Customer\Block\Adminhtml\Edit\GenericButton;
use Magento\Customer\Model\ResourceModel\CustomerRepository;
use Magento\Framework\Registry;
use Magento\Framework\View\Element\UiComponent\Control\ButtonProviderInterface;
use Magento\Store\Model\Website;

class LoginButton extends GenericButton implements ButtonProviderInterface
{
    public const ADMIN_RESOURCE = 'Amasty_CustomerLogin::admin_login';

    /**
     * @var CustomerRepository
     */
    private $customerRepository;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var bool
     */
    private $isAllowed = true;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var string
     */
    private $openNewTab = '';

    public function __construct(
        Context $context,
        Registry $registry,
        CustomerRepository $customerRepository,
        ConfigProvider $configProvider
    ) {
        parent::__construct($context, $registry);
        $this->isAllowed = $context->getAuthorization()->isAllowed(self::ADMIN_RESOURCE);
        $this->customerRepository = $customerRepository;
        $this->storeManager = $context->getStoreManager();
        $this->scopeConfig = $context->getScopeConfig();
        $this->configProvider = $configProvider;
    }

    /**
     * @return array
     */
    public function getButtonData(): array
    {
        $data = [];

        try {
            if ($this->isAllowed && $customerId = $this->getCustomerId()) {
                $this->openNewTab = $this->configProvider->isNewTab() ? ", '_blank'" : ", '_self'";
                $customer = $this->customerRepository->getById($customerId);
                $customerWebsiteId = (int)$customer->getWebsiteId();
                $customerStoreId = (int)$customer->getStoreId();

                if ($this->scopeConfig->getValue('customer/account_share/scope')) {
                    if (!$this->configProvider->isAllowSelectStore()) {
                        $data = [
                            'label' => __('Login as Customer'),
                            'class' => 'amasty-customer-login',
                            'on_click' =>
                                sprintf("window.open('%s'%s);", $this->getLoginLink(0, 0), $this->openNewTab),
                            'sort_order' => 62,
                        ];
                    } else {
                        $data = $this->getSplitButtonData();
                        $website = $this->storeManager->getWebsite($customerWebsiteId);
                        $data['options'] = $this->getStoresDataPerWebsite($website, $customerStoreId, []);
                    }
                } else {
                    $data = $this->getSplitButtonData();
                    $data['options'] = $this->getWebsitesData($customerWebsiteId, $customerStoreId);
                }
            }
        } catch (\Exception $e) {
            null;
        }

        return $data;
    }

    /**
     * @param int $website
     * @param int $storeId
     *
     * @return string
     */
    private function getLoginLink(int $websiteId, int $storeId): string
    {
        $data = ['customer_id' => $this->getCustomerId()];

        if ($websiteId) {
            $data['website'] = $websiteId;
        }

        if ($storeId) {
            $data['store'] = $storeId;
        }

        return $this->getUrl('amcustomerlogin/customer/login', $data);
    }

    /**
     * @return array
     */
    private function getSplitButtonData(): array
    {
        return [
            'label' => __('Login as Customer'),
            'class_name' => SplitButton::class,
            'button_class' => 'amasty-customer-login',
            'options' => []
        ];
    }

    /**
     * @param string $label
     * @param bool   $default
     * @param int    $sortOrder
     * @param int    $websiteId
     * @param int    $storeId
     *
     * @return array
     */
    private function getOptionData(string $label, bool $default, int $sortOrder, int $websiteId, int $storeId): array
    {
        return [
            'label' => $label,
            'default' => (int)$default,
            'onclick' => sprintf(
                "window.open('%s'%s);",
                $this->getLoginLink($websiteId, $storeId),
                $this->openNewTab
            ),
            'sort_order' => $sortOrder,
        ];
    }

    /**
     * @param Website $website
     * @param int     $customerStoreId
     * @param array   $data
     *
     * @return array
     */
    private function getStoresDataPerWebsite(Website $website, int $customerStoreId, array $data): array
    {
        try {
            foreach ($website->getStores() as $storeId => $store) {
                $data[] = $this->getOptionData(
                    __('Login on `%1` (%2)', $store->getName(), $website->getName())->render(),
                    $storeId === $customerStoreId,
                    (int)$storeId,
                    (int)$website->getId(),
                    (int)$storeId
                );
            }
        } catch (\Exception $e) {
            null;
        }

        return $data;
    }

    /**
     * @param int $customerWebsiteId
     * @param int $customerStoreId
     *
     * @return array
     */
    private function getWebsitesData(int $customerWebsiteId, int $customerStoreId): array
    {
        $data = [];
        $allowSelectStore = $this->configProvider->isAllowSelectStore();

        try {
            foreach ($this->storeManager->getWebsites() as $websiteId => $website) {
                if ($allowSelectStore) {
                    $data = $this->getStoresDataPerWebsite($website, $customerStoreId, $data);
                } else {
                    $data[] = $this->getOptionData(
                        __('Login on Website "%1"', $website->getName())->render(),
                        $websiteId === $customerWebsiteId,
                        (int)$websiteId,
                        (int)$websiteId,
                        0
                    );
                }
            }
        } catch (\Exception $e) {
            null;
        }

        return $data;
    }
}
