<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Custom Stock Status Countdown for Magento 2 (System)
*/

declare(strict_types=1);

namespace Amasty\CustomStockStatusTimer\Observer\Backend\Rule\Form;

use Amasty\CustomStockStatusTimer\Api\Data\StockstatusTimerInformationInterface;
use Amasty\CustomStockStatusTimer\Api\StockstatusTimerInformation\RemoveByStockstatusRuleIdInterface;
use Amasty\CustomStockStatusTimer\Api\StockstatusTimerInformation\SaveInterface;
use Amasty\CustomStockStatusTimer\Model\Source\CountdownTimers;
use Amasty\CustomStockStatusTimer\Model\StockstatusTimerInformation\LoadByRuleIdOrGetNew;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;

class ProcessSaveTimerInfo implements ObserverInterface
{
    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var SaveInterface
     */
    private $saveStockstatusTimerInfo;

    /**
     * @var LoadByRuleIdOrGetNew
     */
    private $loadByRuleIdOrGetNew;

    /**
     * @var RemoveByStockstatusRuleIdInterface
     */
    private $removeByStockstatusRuleId;

    public function __construct(
        RequestInterface $request,
        SaveInterface $saveStockstatusTimerInfo,
        LoadByRuleIdOrGetNew $loadByRuleIdOrGetNew,
        RemoveByStockstatusRuleIdInterface $removeByStockstatusRuleId
    ) {
        $this->request = $request;
        $this->saveStockstatusTimerInfo = $saveStockstatusTimerInfo;
        $this->loadByRuleIdOrGetNew = $loadByRuleIdOrGetNew;
        $this->removeByStockstatusRuleId = $removeByStockstatusRuleId;
    }

    public function execute(Observer $observer)
    {
        /** @var \Amasty\Stockstatus\Model\Rule $rule **/
        $rule = $observer->getEvent()->getObject();
        $countdownData = $this->getCountdownData();

        if ($rule && $countdownData !== null) {
            $ruleId = (int)$rule->getId();

            if ($countdownData[StockstatusTimerInformationInterface::TIMER_ID] === CountdownTimers::UNSELECTED_VALUE) {
                $this->removeTimerInfo($ruleId);
            } else {
                $this->saveTimerInfo($ruleId, $countdownData);
            }
        }
    }

    private function removeTimerInfo(int $ruleId): void
    {
        $this->removeByStockstatusRuleId->execute($ruleId);
    }

    private function saveTimerInfo(int $ruleId, array $countdownData): void
    {
        $stockstatusTimerInfoModel = $this->loadByRuleIdOrGetNew->execute($ruleId);
        $stockstatusTimerInfoModel->addData($countdownData);
        $stockstatusTimerInfoModel->setStockstatusRuleId($ruleId);
        $this->saveStockstatusTimerInfo->execute($stockstatusTimerInfoModel);
    }

    private function getCountdownData(): ?array
    {
        $ruleData = $this->request->getParam('rule', []);
        $countdownWidgetId = $ruleData[StockstatusTimerInformationInterface::TIMER_ID] ?? false;
        $countdownPosition = $ruleData[StockstatusTimerInformationInterface::TIMER_POSITION] ?? false;

        if ($countdownWidgetId !== false && $countdownPosition !== false) {
            return [
                StockstatusTimerInformationInterface::TIMER_ID => (int)$countdownWidgetId,
                StockstatusTimerInformationInterface::TIMER_POSITION => (int)$countdownPosition
            ];
        }

        return null;
    }
}
