<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Custom Stock Status Countdown for Magento 2 (System)
*/

declare(strict_types=1);

namespace Amasty\CustomStockStatusTimer\Observer;

use Amasty\CustomStockStatusTimer\Api\Data\StockstatusTimerInformationInterface;
use Amasty\CustomStockStatusTimer\Api\StockstatusTimerInformation\GetByStockstatusRuleIdInterface;
use Amasty\CustomStockStatusTimer\Model\CountdownTimer\GetCountdownBlockByWidgetId;
use Amasty\CustomStockStatusTimer\Model\Source\TimerPositions;
use Amasty\Stockstatus\Api\Rule\GetByProductIdAndStoreIdInterface;
use Amasty\Stockstatus\Block\CustomStockStatus;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;

class AddCountdownTimerInfo implements ObserverInterface
{
    /**
     * @var GetByProductIdAndStoreIdInterface
     */
    private $getByProductIdAndStoreId;

    /**
     * @var GetByStockstatusRuleIdInterface
     */
    private $getByStockstatusRuleId;

    /**
     * @var GetCountdownBlockByWidgetId
     */
    private $getCountdownBlockByWidgetId;

    public function __construct(
        GetByProductIdAndStoreIdInterface $getByProductIdAndStoreId,
        GetByStockstatusRuleIdInterface $getByStockstatusRuleId,
        GetCountdownBlockByWidgetId $getCountdownBlockByWidgetId
    ) {
        $this->getByProductIdAndStoreId = $getByProductIdAndStoreId;
        $this->getByStockstatusRuleId = $getByStockstatusRuleId;
        $this->getCountdownBlockByWidgetId = $getCountdownBlockByWidgetId;
    }

    public function execute(Observer $observer)
    {
        /** @var CustomStockStatus $stockStatusBlock **/
        $stockStatusBlock = $observer->getEvent()->getStockstatusBlock();

        if ($stockStatusBlock
            && !$stockStatusBlock->isInProductList()
            && $stockStatusBlock->hasStockstatusInformation()
        ) {
            $product = $stockStatusBlock->getProduct();
            $stockstatusTimerInfo = $this->getStockstatusTimerInfoByProduct($product);

            if ($stockstatusTimerInfo) {
                $countdownBlock = $this->getCountdownBlockByWidgetId->execute($stockstatusTimerInfo->getTimerId());

                if ($countdownBlock) {
                    $stockStatusBlock->setChild(
                        $this->getStockstatusTimerPosition($stockstatusTimerInfo->getTimerPosition()),
                        $countdownBlock
                    );
                }
            }
        }
    }

    private function getStockstatusTimerInfoByProduct(ProductInterface $product): ?StockstatusTimerInformationInterface
    {
        $stockstatusRule = $this->getByProductIdAndStoreId->execute(
            (int)$product->getId(),
            (int)$product->getStoreId()
        );

        if ($stockstatusRule) {
            try {
                $stockstatusTimerInfo = $this->getByStockstatusRuleId->execute((int)$stockstatusRule->getId());
            } catch (NoSuchEntityException $e) {
                $stockstatusTimerInfo = null;
            }
        } else {
            $stockstatusTimerInfo = null;
        }

        return $stockstatusTimerInfo;
    }

    private function getStockstatusTimerPosition(int $positionIdentifier): string
    {
        return $positionIdentifier === TimerPositions::POSITION_BEFORE_STOCKSTATUS
            ? CustomStockStatus::BEFORE_CONTAINER
            : CustomStockStatus::AFTER_CONTAINER;
    }
}
