<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Custom Form Compatibility with Hyva Theme
 */

namespace Amasty\CustomFormHyva\ViewModel;

use DateTime;
use Amasty\Customform\Model\ConfigProvider;
use Amasty\Customform\Model\SurveyAvailableResolver;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Amasty\Customform\Model\FormRepository;
use Amasty\Customform\Api\Data\FormInterface;
use Amasty\Customform\Model\Form as AmastyForm;
use Magento\Framework\Exception\NoSuchEntityException;

class Form implements ArgumentInterface
{
    /**
     * @var FormRepository
     */
    protected $formRepository;

    /**
     * @var SurveyAvailableResolver
     */
    protected $surveyAvailableResolver;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * Form constructor.
     * @param ConfigProvider $configProvider
     * @param FormRepository $formRepository
     * @param SurveyAvailableResolver $surveyAvailableResolver
     */
    public function __construct(
        ConfigProvider $configProvider,
        FormRepository $formRepository,
        SurveyAvailableResolver $surveyAvailableResolver
    ) {
        $this->formRepository = $formRepository;
        $this->configProvider = $configProvider;
        $this->surveyAvailableResolver = $surveyAvailableResolver;
    }

    /**
     * @param $formId
     * @return FormInterface|AmastyForm|mixed
     * @throws NoSuchEntityException
     */
    public function getForm($formId)
    {
        return $this->formRepository->get($formId);
    }

    public function isSurveyAvailable(int $formId): ?bool
    {
        return $formId ? $this->surveyAvailableResolver->isSurveyAvailable($formId) : true;
    }

    public function getPlaceholder($field): string
    {
        return isset($field['placeholder']) && $field['placeholder']
            ? ('placeholder="' . $field['placeholder'] . '" ') : ' ';
    }

    public function getErrorMessage($field): string
    {
        $errorMsgPrefix = 'data-errormsg="';
        $errorMessage = isset($field['errorMessage']) && $field['errorMessage']
            ? ($errorMsgPrefix . $field['errorMessage'] . '" ') : ' ';

        if (isset($field['validation']) && $field['validation'] === 'validate-alpha') {
            $errorMessage = $errorMsgPrefix . __('Please use letters only (a-z or A-Z) in this field.') . '"';
        }

        if (isset($field['validation']) && $field['validation'] === 'validate-alphanum') {
            $errorMessage = $errorMsgPrefix
                . __('Please use only letters (a-z or A-Z) or numbers (0-9) in this field.')
                . __('No spaces or other characters are allowed.') . '"';
        }

        return $errorMessage;
    }

    public function getRegExp($field): string
    {
        return isset($field['regexp']) && $field['regexp']
            ? ('pattern="' . $field['regexp'] . '" ') : ' ';
    }

    public function getMaxLength($field): string
    {
        return isset($field['maxlength']) && $field['maxlength']
            ? ('maxlength="' . $field['maxlength'] . '" ') : ' ';
    }

    public function getMinLength($field): string
    {
        return isset($field['minlength']) && $field['minlength']
            ? ('minlength="' . $field['minlength'] . '" ') : ' ';
    }

    public function getMax($field): string
    {
        return isset($field['max']) && $field['max']
            ? ('max="' . $field['max'] . '" ') : ' ';
    }

    public function getMin($field): string
    {
        return isset($field['min']) && $field['min']
            ? ('min="' . $field['min'] . '" ') : ' ';
    }

    public function getStep($field): string
    {
        return isset($field['step']) && $field['step']
            ? ('step="' . $field['step'] . '" ') : ' ';
    }

    public function getValue($field): ?string
    {
        $value = isset($field['value']) && $field['value'] ? $field['value'] : null;
        if ($value && $field['type'] === 'date') {
            $date = DateTime::createFromFormat($this->configProvider->getDateFormat(), $value);

            if (!$date) {
                $frontDateFormat = preg_replace("/y/", 'yy', $this->configProvider->getDateFormat());
                $date = DateTime::createFromFormat($frontDateFormat, $value);
            }

            if (!$date) {
                $replace = [
                    '/dd/i' => 'd',
                    '/mm/i' => 'm',
                    '/yy/' => 'Y',
                    '/YY/' => 'Y'
                ];
                $frontDateFormat = preg_replace(array_keys($replace), array_values($replace), $frontDateFormat);
                $date = DateTime::createFromFormat($frontDateFormat, $value);
            }

            if ($date) {
                $value = $date->format('Y-m-d');
            }
        }

        return $value;
    }

    public function getValueString($field): string
    {
        $value = $this->getValue($field);
        return $value ? ('value="' . $value . '" ') : ' ';
    }

    public function getAllowedExtension($field): string
    {
        return isset($field['allowed_extension']) && $field['allowed_extension']
            ? ('accept="' . $field['allowed_extension'] . '" ') : ' ';
    }

    public function getAlfa($field): string
    {
        return isset($field['validation']) && $field['validation'] === 'validate-alpha'
            ? 'pattern="[a-zA-Z\'-\'\s]*" ' : ' ';
    }

    public function getAlfaNum($field): string
    {
        return isset($field['validation']) && $field['validation'] === 'validate-alphanum'
            ? 'pattern="^[a-zA-Z0-9 ]+$" ' : ' ';
    }

    public function getMultiple($field): string
    {
        return isset($field['multiple']) && $field['multiple'] ? 'multiple ' : ' ';
    }

    public function getRequired($field): string
    {
        return isset($field['required']) && $field['required'] ? 'required ' : ' ';
    }

    public function getFormFieldValidation($field): string
    {
        $fieldValidation = '';

        if ($field) {
            $errorMessage = $this->getErrorMessage($field);
            $placeholder = $this->getPlaceholder($field);
            $maxlength = $this->getMaxLength($field);
            $minlength = $this->getMinLength($field);
            $regexp = $this->getRegExp($field);
            $max = $this->getMax($field);
            $min = $this->getMin($field);
            $step = $this->getStep($field);
            $value = $this->getValueString($field);
            $alpha = $this->getAlfa($field);
            $alphanum = $this->getAlfaNum($field);
            $multiple = $this->getMultiple($field);
            $required = $this->getRequired($field);
            $allowed_extension = $this->getAllowedExtension($field);

            $fieldValidation =
                $placeholder . $errorMessage .
                $regexp . $maxlength .
                $minlength . $max . $min .
                $step . $value . $alpha . $alphanum .
                $allowed_extension . $multiple . $required;
        }

        return $fieldValidation;
    }

    public function getFieldInputType($field): string
    {
        $attrPrefix = 'type=';
        $inputType = 'text';
        $attrPostfix = '';

        switch ($field['type']) {
            case 'number':
                $inputType = 'number';
                break;
            case 'date':
                $inputType = 'date';
                break;
            case 'time':
                $inputType = 'time';
                break;
            case 'file':
                $inputType = 'file';
                break;
        }

        if (isset($field['validation'])
            && $field['validation'] === 'validate-email') {
            $inputType = 'email';
        }

        if (isset($field['validation'])
            && $field['validation'] === 'validate-url') {
            $inputType = 'url';
        }

        if (isset($field['validation'])
            && $field['validation'] === 'validate-number') {
            $inputType = 'number';
            $attrPostfix = 'step="any"';
        }

        if (isset($field['validation'])
            && $field['validation'] === 'validate-digits') {
            $inputType = 'number';
        }

        return $attrPrefix . '"' . $inputType . '" ' . $attrPostfix;
    }

    public function getDepFormFieldName($val): string
    {
        $depFieldName = '';

        if ($val) {
            $depFieldName = trim(str_replace(['-', '.',' ','_'], '', $val));
        }

        return $depFieldName;
    }
}
