/**
 *  Amasty Custom Button widget
 */

define([
    'jquery'
], function ($) {
    'use strict';

    $.widget('am.customButton', {
        options: {
            text: '',
            hoverText: '',
            animation: '',
            colors: {
                background: '',
                hoverBackground: '',
                borderColor: ''
            }
        },
        unitPostfix: 'px',
        intersectionObserver: null,
        intersectionObserverName: 'IntersectionObserver',
        classes: {
            hidden: '-am-hidden',
            animationPrefix: '-animation-',
            animationElement: 'amcustombutton-animation-element'
        },
        selectors: {
            button: '[data-amcustombutton-js="button"]',
            buttonTextBlock: '[data-amcustombutton-js="button-text"]'
        },

        /**
         * @private
         * @returns {void}
         */
        _create: function () {
            this._initNodes();
            this._initLoadObserver();
            this._changeTextListener();
        },

        /**
         * @private
         * @returns {void}
         */
        _initNodes: function () {
            this.button = this.element.find(this.selectors.button);
            this.buttonTextBlock = this.element.find(this.selectors.buttonTextBlock);
            this.animatingBlock = $('<i></i>', {
                class: this.classes.animationElement
            });
        },

        /**
         *  Change button text on hover
         * @private
         * @returns {void}
         */
        _changeTextListener: function () {
            if (!this.options.hoverText) {
                return;
            }

            $(this.button).hover(
                this._setButtonText.bind(this, this.options.hoverText),
                this._setButtonText.bind(this, this.options.text)
            );
        },

        /**
         * @param value
         * @private
         * @return {void}
         */
        _setButtonText: function (value) {
            this.buttonTextBlock.text(value);
        },

        /**
         * @private
         * @returns {void}
         */
        _setButtonWidth: function () {
            this.button.width(this._getButtonWidth() + this.unitPostfix);
        },

        /**
         * @private
         * @returns {void}
         */
        _setButtonHeight: function () {
            var button = this.button[0];

            button.style.minHeight = button.offsetHeight + this.unitPostfix;
        },

        /**
         * @private
         * @returns {void}
         */
        _setButtonSize: function () {
            if (!this.options.hoverText) {
                return;
            }

            if (this.options.hoverText.length >= this.options.text.length) {
                this._setButtonText(this.options.hoverText);
                this._setButtonWidth();
                this._setButtonHeight();
                this._setButtonText(this.options.text);
            } else {
                this._setButtonWidth();
                this._setButtonHeight();
            }
        },

        /**
         * IntersectionObserver polyfill for IE11
         *
         * @private
         * @returns {void}
         */
        _buttonLoadingObserverPolyfill: function () {
            var self = this;

            self.polyfilIinterval = setInterval(function () {
                if (self.button[0].offsetParent) {
                    self._setButtonSize();
                    self._setAnimationEffect(self.options.animation);
                    self._showButton();
                    clearInterval(self.polyfilIinterval);
                }
            }, 500);
        },

        /**
         * @private
         * @returns {void}
         */
        _buttonLoadingObserver: function () {
            var self = this,
                element,
                options = {
                    root: null,
                    rootMargin: '50px',
                    threshold: 0
                };

            this.intersectionObserver = new IntersectionObserver(function (entries) {
                entries.forEach(function (entry) {
                    if (entry.isIntersecting) {
                        self._setButtonSize();
                        self._setAnimationEffect(self.options.animation);
                        self._showButton();
                        self.intersectionObserver.unobserve(entry.target);
                    }
                });
            }, options);

            element = this.element.context ? this.element.context.firstElementChild : this.element[0];
            this.intersectionObserver.observe(element);
        },

        /**
         * @private
         * @returns {void}
         */
        _initLoadObserver: function () {
            // eslint-disable-next-line no-unused-expressions
            this.intersectionObserverName in window
                ? this._buttonLoadingObserver()
                : this._buttonLoadingObserverPolyfill();
        },

        /**
         * @private
         * @returns {Number}
         */
        _getButtonWidth: function () {
            return this.button[0].offsetWidth;
        },

        /**
         * Get x and y dimensions of target element
         * @private
         * @param element {object}
         * @param event {object}
         * @returns {Object}
         */
        _getDimensions: function (element, event) {
            var parentOffset = element.offset(),
                relX = event.pageX - parentOffset.left,
                relY = event.pageY - parentOffset.top;

            return { x: relX, y: relY };
        },

        /**
         * @param {String} effect - animation effect
         * @private
         * @returns {void}
         */
        _setAnimationEffect: function (effect) {
            this.button.addClass(this.classes.animationPrefix + effect);

            switch (effect) {
                case 'circle':
                    this._animationCircleOverlap();
                    break;
                case 'position-aware':
                    this._animationPositionAware();
                    break;
                case 'collision':
                    this._animationCollision();
                    break;
                case 'border-race':
                    this._animationBorderRace();
                    break;
                default:
            }
        },

        /**
         * @private
         * @returns {void}
         */
        _showButton: function () {
            this.element.removeClass(this.classes.hidden);
        },

        /**
         * @private
         * @returns {void}
         */
        _animationCircleOverlap: function () {
            var self = this,
                buttonSize = self._getButtonWidth(),
                circleBlock = this.animatingBlock.clone(),
                shadowValue = self.options.colors.hoverBackground + ' inset 0 0 0 ' + buttonSize + self.unitPostfix,
                animatingElement;

            this.button.css('backgroundColor', this.options.colors.background).append(circleBlock);

            animatingElement = this.button.find(circleBlock);

            animatingElement.css({
                'width': buttonSize * 2 + this.unitPostfix,
                'height': buttonSize * 2 + this.unitPostfix,
                'left': '-' + buttonSize / 2 + this.unitPostfix
            });

            this.button.hover(function () {
                animatingElement.css('boxShadow', shadowValue);
            }, function () {
                animatingElement.css('boxShadow', 'none');
            });
        },

        /**
         * @private
         * @returns {void}
         */
        _animationPositionAware: function () {
            var self = this,
                buttonSize = self._getButtonWidth(),
                circleBlock = this.animatingBlock.clone(),
                animatingElement,
                dimensions;

            this.button.css('backgroundColor', this.options.colors.background).append(circleBlock);

            animatingElement = this.button.find(circleBlock);

            animatingElement.css('backgroundColor', this.options.colors.hoverBackground);

            this.button.hover(function (event) {
                dimensions = self._getDimensions($(this), event);

                animatingElement.css({
                    top: dimensions.y,
                    left: dimensions.x,
                    width: buttonSize * 2.25 + self.unitPostfix,
                    height: buttonSize * 2.25 + self.unitPostfix
                });
            }, function (event) {
                dimensions = self._getDimensions($(this), event);

                animatingElement.css({
                    top: dimensions.y,
                    left: dimensions.x,
                    width: 0,
                    height: 0
                });
            });
        },

        /**
         * @private
         * @returns {void}
         */
        _animationCollision: function () {
            var self = this,
                buttonSize = self._getButtonWidth(),
                hoverBackgroundColor = this.options.colors.hoverBackground,
                animatingElementSize = 20,
                transitionDelay = 400,
                animatingElements,
                timeout;

            this.button
                .css('backgroundColor', this.options.colors.background)
                .prepend(this.animatingBlock.clone())
                .append(this.animatingBlock.clone());

            animatingElements = this.button.find('.' + this.classes.animationElement);

            animatingElements.css('backgroundColor', hoverBackgroundColor);

            this.button.hover(function () {
                timeout = setTimeout(function () {
                    animatingElements.css({
                        width: buttonSize * 1.5 + self.unitPostfix,
                        height: buttonSize * 1.5 + self.unitPostfix
                    });
                }, transitionDelay);
            }, function () {
                clearTimeout(timeout);

                animatingElements.css({
                    width: animatingElementSize + self.unitPostfix,
                    height: animatingElementSize + self.unitPostfix
                });
            });
        },

        /**
         * @private
         * @returns {void}
         */
        _animationBorderRace: function () {
            this.button
                .css({ 'border': 0, 'borderRadius': 0 })
                .prepend(this.animatingBlock.clone(), this.animatingBlock.clone())
                .append(this.animatingBlock.clone(), this.animatingBlock.clone());

            this.button
                .find('.' + this.classes.animationElement)
                .css('backgroundColor', this.options.colors.borderColor);
        }
    });

    return $.am.customButton;
});
